const express = require('express');
const cors = require('cors');
const axios = require('axios');
require('dotenv').config();

const app = express();
const PORT = process.env.PORT || 3000;
const COZE_TOKEN = process.env.COZE_TOKEN;

// Middleware
app.use(cors());
app.use(express.json());

// 健康检查端点
app.get('/', (req, res) => {
  res.send('OOTD Mini Program Server is running');
});

// 天气API端点
app.get('/api/weather', async (req, res) => {
  try {
    const { city = '北京' } = req.query;
    
    const response = await axios.post('https://api.coze.cn/v1/workflow/run', {
      parameters: {
        city: city
      },
      workflow_id: "7479466440060878857",
      app_id: "7479438233773162508"
    }, {
      headers: {
        'Authorization': `Bearer ${COZE_TOKEN}`,
        'Content-Type': 'application/json'
      }
    });

    // 从字符串解析数据为JSON
    if (response.data.code === 0) {
      const weatherData = JSON.parse(response.data.data);
      console.log(`Successfully fetched weather data for ${city}`);
      return res.json({ 
        success: true, 
        data: weatherData.output || [],
        city: city
      });
    } else {
      throw new Error(`Failed to fetch weather data for ${city}`);
    }
  } catch (error) {
    console.error('Weather API Error:', error);
    res.status(500).json({ 
      success: false, 
      error: 'Failed to fetch weather data' 
    });
  }
});

// OOTD生成API端点
app.post('/api/generate-ootd', async (req, res) => {
  try {
    const { city, gender, personalTrait, selectedStyle, weather } = req.body;
    
    if (!city || !gender || !selectedStyle || !weather) {
      return res.status(400).json({
        success: false,
        error: '缺少必要参数'
      });
    }
    
    const response = await axios.post('https://api.coze.cn/v1/workflow/run', {
      parameters: {
        city,
        gender,
        personalTrait: personalTrait || '',
        selectedStyle,
        weather
      },
      workflow_id: "7479471711643025420",
      app_id: "7479438233773162508"
    }, {
      headers: {
        'Authorization': `Bearer ${COZE_TOKEN}`,
        'Content-Type': 'application/json'
      }
    });
    
    if (response.data.code === 0) {
      const ootdData = JSON.parse(response.data.data);
      console.log(`Successfully generated OOTD for ${city}, style: ${selectedStyle}`);
      return res.json({
        success: true,
        data: {
          advice: ootdData.advice,
          images: ootdData.output
        }
      });
    } else {
      throw new Error(`Failed to generate OOTD`);
    }
  } catch (error) {
    console.error('OOTD Generation Error:', error);
    res.status(500).json({
      success: false,
      error: '生成穿搭建议失败'
    });
  }
});

// 启动服务器
app.listen(PORT, () => {
  console.log(`服务器已在端口 ${PORT} 上运行`);
}); 