"use client"

import { useState } from "react"
import { Textarea } from "@/components/ui/textarea"
import { Button } from "@/components/ui/button"
import {
  Drawer,
  DrawerContent,
  DrawerHeader,
  DrawerTitle,
  DrawerClose,
} from "@/components/ui/drawer"
import { ReportDisplay } from "@/components/report-display"
import { ReportSkeleton } from "@/components/report-skeleton"
import { AnalysisReport } from "@/lib/mock-data"
import { X } from "lucide-react"

const MAX_CHARACTERS = 8000

export default function Home() {
  const [inputText, setInputText] = useState("")
  const [isOpen, setIsOpen] = useState(false)
  const [isLoading, setIsLoading] = useState(false)
  const [report, setReport] = useState<AnalysisReport | null>(null)
  const [error, setError] = useState<string | null>(null)

  const remainingCharacters = MAX_CHARACTERS - inputText.length

  const validateInput = (text: string): boolean => {
    if (!text.trim()) {
      setError("请输入内容")
      return false
    }
    
    const chineseEnglishPattern = /[\u4e00-\u9fa5a-zA-Z\s]/g
    const validCharacters = (text.match(chineseEnglishPattern) || []).length
    const invalidRatio = 1 - validCharacters / text.length
    
    if (invalidRatio > 0.5) {
      setError("输入内容包含过多无意义字符，请检查后重试")
      return false
    }
    
    return true
  }

  const handleAnalyze = async () => {
    if (!validateInput(inputText)) {
      return
    }

    setError(null)
    setIsOpen(true)
    setIsLoading(true)
    setReport(null)

    try {
      const response = await fetch("/api/report", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({ text: inputText }),
      })

      let data
      try {
        data = await response.json()
      } catch (parseError) {
        throw new Error("服务器响应格式错误")
      }

      if (!response.ok) {
        throw new Error(data.error || "分析失败，请稍后重试")
      }

      setReport(data as AnalysisReport)
    } catch (err) {
      let errorMessage = "分析失败，请稍后重试"
      
      if (err instanceof Error) {
        errorMessage = err.message
      } else if (typeof err === "string") {
        errorMessage = err
      }
      
      setError(errorMessage)
    } finally {
      setIsLoading(false)
    }
  }

  const handleRetry = () => {
    setError(null)
    handleAnalyze()
  }

  return (
    <div className="relative min-h-screen overflow-hidden bg-[#f3f0ff] text-black transition-colors duration-300 dark:bg-[#050505] dark:text-white">
      <div className="pointer-events-none absolute inset-0">
        <div className="absolute -left-24 top-24 hidden h-64 w-64 rotate-6 border-4 border-black bg-[#ffce52] shadow-[18px_18px_0_0_#000] dark:border-white dark:shadow-[18px_18px_0_0_rgba(255,255,255,0.16)] lg:block" />
        <div className="absolute -right-20 bottom-20 h-72 w-72 -rotate-12 border-4 border-black bg-[#8be7ff] shadow-[20px_20px_0_0_#000] dark:border-white dark:shadow-[20px_20px_0_0_rgba(255,255,255,0.12)]" />
        <div className="absolute inset-x-0 top-0 h-32 bg-gradient-to-b from-[#ff90e8]/40 via-transparent to-transparent dark:from-[#f97316]/25" />
        <div className="absolute inset-x-0 bottom-0 h-40 bg-gradient-to-t from-[#f7ff8d]/50 via-transparent to-transparent dark:from-[#22d3ee]/20" />
      </div>

      <main className="relative z-10 container mx-auto max-w-4xl px-6 py-24">
        <div className="space-y-12">
          <section className="relative overflow-hidden rounded-none border-4 border-black bg-[#fff4d6] p-12 shadow-[20px_20px_0_0_#000] transition-transform duration-200 dark:border-white dark:bg-[#131313] dark:shadow-[20px_20px_0_0_rgba(255,255,255,0.12)]">
            <div className="absolute -right-10 top-14 hidden h-40 w-40 rotate-12 border-4 border-black bg-white/50 shadow-[12px_12px_0_0_#000] dark:border-white/70 dark:bg-white/10 dark:shadow-[12px_12px_0_0_rgba(255,255,255,0.1)] sm:block" />
            <div className="flex flex-wrap items-center justify-between gap-4">
              <span className="rounded-full border-2 border-black bg-white px-4 py-1 text-xs font-black uppercase tracking-[0.25em] shadow-[6px_6px_0_0_#000] dark:border-white dark:bg-[#090909] dark:shadow-[6px_6px_0_0_rgba(255,255,255,0.12)]">
                AI Toolkit
              </span>
              <span className="rounded-full border-2 border-black bg-[#8be7ff] px-3 py-1 text-xs font-bold uppercase tracking-widest text-black shadow-[6px_6px_0_0_#000] dark:border-white dark:bg-[#2563eb] dark:text-white dark:shadow-[6px_6px_0_0_rgba(255,255,255,0.16)]">
                全量中文分析
              </span>
            </div>
            <h1 className="mt-10 max-w-3xl text-5xl font-black leading-snug tracking-tight text-black sm:text-6xl dark:text-white">
              AI初创项目评估工具
            </h1>
            <p className="mt-6 max-w-2xl text-lg font-medium text-black/80 dark:text-white/80">
              快速从初创公司落地页文案中生成评估报告，覆盖核心信息提炼、投资价值判断与风险提示，以高对比的可视化呈现帮助你迅速决策。
            </p>
            <div className="mt-10 grid gap-5 sm:grid-cols-3">
              <div className="rounded-none border-2 border-black bg-white px-4 py-5 shadow-[10px_10px_0_0_#000] dark:border-white dark:bg-[#0a0a0a] dark:shadow-[10px_10px_0_0_rgba(255,255,255,0.14)]">
                <p className="text-xs font-black uppercase tracking-[0.3em] text-[#f97316] dark:text-[#fbbf24]">
                  03 模块
                </p>
                <p className="mt-3 text-sm font-semibold">
                  核心信息、价值评估、风险提示一键生成
                </p>
              </div>
              <div className="rounded-none border-2 border-black bg-[#f7ff8d] px-4 py-5 shadow-[10px_10px_0_0_#000] dark:border-white dark:bg-[#1f2937] dark:shadow-[10px_10px_0_0_rgba(255,255,255,0.14)]">
                <p className="text-xs font-black uppercase tracking-[0.3em] text-black dark:text-white">
                  120 秒
                </p>
                <p className="mt-3 text-sm font-semibold">
                  两分钟内完成分析并输出结构化报告
                </p>
              </div>
              <div className="rounded-none border-2 border-black bg-white px-4 py-5 shadow-[10px_10px_0_0_#000] dark:border-white dark:bg-[#0a0a0a] dark:shadow-[10px_10px_0_0_rgba(255,255,255,0.14)]">
                <p className="text-xs font-black uppercase tracking-[0.3em] text-[#2563eb]">
                  风险护栏
                </p>
                <p className="mt-3 text-sm font-semibold">
                  自动识别噪点与无效信息，提示潜在风险
                </p>
              </div>
            </div>
          </section>

          <section className="rounded-none border-4 border-black bg-white p-10 shadow-[20px_20px_0_0_#000] dark:border-white dark:bg-[#0a0a0a] dark:shadow-[20px_20px_0_0_rgba(255,255,255,0.1)]">
            <div className="flex flex-wrap items-center justify-between gap-3 border-b-4 border-black pb-6 dark:border-white">
              <h2 className="text-2xl font-black uppercase tracking-[0.2em] text-black dark:text-white">
                文案输入面板
              </h2>
              <span className="rounded-none border-2 border-black bg-[#ff90e8] px-3 py-1 text-xs font-black uppercase tracking-[0.4em] text-black shadow-[6px_6px_0_0_#000] dark:border-white dark:bg-[#f97316] dark:text-white dark:shadow-[6px_6px_0_0_rgba(255,255,255,0.12)]">
                步骤 01
              </span>
            </div>

            <div className="mt-6 space-y-4">
              <Textarea
                placeholder="请将初创公司落地页的全部文本复制到这里..."
                value={inputText}
                onChange={(event) => {
                  const inputValue = event.target.value
                  if (inputValue.length <= MAX_CHARACTERS) {
                    setInputText(inputValue)
                    setError(null)
                  }
                }}
                className="min-h-[220px] max-h-[480px] rounded-none border-4 border-black bg-white p-4 text-base font-medium text-black shadow-[12px_12px_0_0_#000] transition-transform duration-150 focus-visible:-translate-y-1 focus-visible:border-black focus-visible:outline-none focus-visible:ring-0 focus-visible:shadow-[16px_16px_0_0_#000] dark:border-white dark:bg-[#050505] dark:text-white dark:shadow-[12px_12px_0_0_rgba(255,255,255,0.12)] dark:focus-visible:shadow-[16px_16px_0_0_rgba(255,255,255,0.16)]"
                rows={8}
              />
              <div className="flex flex-col gap-3 text-sm md:flex-row md:items-center md:justify-between">
                <span className="rounded-none border-2 border-black bg-[#f7ff8d] px-3 py-1 font-semibold text-black shadow-[6px_6px_0_0_#000] dark:border-white dark:bg-[#1f2937] dark:text-white dark:shadow-[6px_6px_0_0_rgba(255,255,255,0.12)]">
                  {remainingCharacters >= 0
                    ? `还可输入 ${remainingCharacters} 字符`
                    : `超出限制 ${Math.abs(remainingCharacters)} 字符`}
                </span>
                {error && (
                  <span className="rounded-none border-2 border-black bg-[#ffb4b4] px-3 py-1 font-semibold text-black shadow-[6px_6px_0_0_#000] dark:border-white dark:bg-[#b91c1c] dark:text-white dark:shadow-[6px_6px_0_0_rgba(255,255,255,0.12)]">
                    {error}
                  </span>
                )}
              </div>
            </div>

            <Button
              onClick={handleAnalyze}
              disabled={
                isLoading ||
                !inputText.trim() ||
                inputText.length > MAX_CHARACTERS
              }
              className="mt-8 w-full rounded-none border-4 border-black bg-[#ffce52] px-8 py-4 text-lg font-black uppercase tracking-[0.3em] text-black shadow-[14px_14px_0_0_#000] transition-transform duration-150 hover:-translate-y-1 hover:shadow-[18px_18px_0_0_#000] disabled:translate-y-0 disabled:cursor-not-allowed disabled:border-black/20 disabled:bg-neutral-200 disabled:text-neutral-500 disabled:shadow-none dark:border-white dark:bg-[#f97316] dark:text-black dark:hover:shadow-[18px_18px_0_0_rgba(249,115,22,0.6)] dark:disabled:border-white/20 dark:disabled:bg-neutral-800 dark:disabled:text-neutral-500 dark:disabled:shadow-none"
              size="lg"
            >
              {isLoading ? "分析中..." : "开始分析"}
            </Button>
          </section>
        </div>
      </main>

      <Drawer open={isOpen} onOpenChange={setIsOpen} direction="top">
        <DrawerContent className="!h-[95vh] !max-h-[95vh] border-t-4 border-black bg-[#fffdf3] shadow-[0_24px_0_0_#000] dark:border-white dark:bg-[#111111] dark:shadow-[0_24px_0_0_rgba(255,255,255,0.12)]">
          <DrawerHeader className="border-b-4 border-black bg-white px-6 py-5 dark:border-white dark:bg-[#0d0d0d]">
            <div className="flex items-center justify-between">
              <DrawerTitle className="text-3xl font-black uppercase tracking-[0.3em] text-black dark:text-white">
                分析结果
              </DrawerTitle>
              <DrawerClose asChild>
                <Button
                  variant="ghost"
                  size="icon"
                  aria-label="关闭分析结果"
                  className="rounded-none border-2 border-black bg-[#ffce52] text-black shadow-[6px_6px_0_0_#000] transition-transform duration-150 hover:-translate-y-0.5 hover:shadow-[10px_10px_0_0_#000] dark:border-white dark:bg-[#1f2937] dark:text-white dark:shadow-[6px_6px_0_0_rgba(255,255,255,0.12)]"
                >
                  <X className="h-4 w-4" />
                </Button>
              </DrawerClose>
            </div>
          </DrawerHeader>
          <div className="flex-1 overflow-y-auto bg-[#fff9e6] p-6 dark:bg-[#141414]">
            {isLoading && (
              <div className="rounded-none border-4 border-black bg-white p-6 shadow-[16px_16px_0_0_#000] dark:border-white dark:bg-[#0b0b0b] dark:shadow-[16px_16px_0_0_rgba(255,255,255,0.12)]">
                <ReportSkeleton />
              </div>
            )}
            {error && (
              <div className="flex h-full flex-col items-center justify-center space-y-4">
                <p className="rounded-none border-2 border-black bg-[#ffb4b4] px-4 py-2 text-lg font-semibold text-black shadow-[8px_8px_0_0_#000] dark:border-white dark:bg-[#b91c1c] dark:text-white dark:shadow-[8px_8px_0_0_rgba(255,255,255,0.12)]">
                  {error}
                </p>
                <Button
                  onClick={handleRetry}
                  className="rounded-none border-4 border-black bg-[#8be7ff] px-8 py-3 text-sm font-black uppercase tracking-[0.4em] text-black shadow-[10px_10px_0_0_#000] transition-transform duration-150 hover:-translate-y-1 hover:shadow-[14px_14px_0_0_#000] dark:border-white dark:bg-[#2563eb] dark:text-white dark:shadow-[10px_10px_0_0_rgba(255,255,255,0.12)]"
                >
                  重试
                </Button>
              </div>
            )}
            {report && !isLoading && (
              <div className="rounded-none border-4 border-black bg-white p-6 shadow-[18px_18px_0_0_#000] dark:border-white dark:bg-[#0b0b0b] dark:shadow-[18px_18px_0_0_rgba(255,255,255,0.12)]">
                <ReportDisplay report={report} />
              </div>
            )}
          </div>
        </DrawerContent>
      </Drawer>
    </div>
  )
}
