import { NextRequest, NextResponse } from "next/server";
import OpenAI from "openai";
import { AnalysisReport } from "@/lib/mock-data";

const MAX_CHARACTERS = 8000;

function getOpenAIClient() {
  const apiKey = process.env.DASHSCOPE_API_KEY;
  if (!apiKey) {
    throw new Error("DASHSCOPE_API_KEY is not configured");
  }
  return new OpenAI({
    apiKey,
    baseURL: "https://dashscope.aliyuncs.com/compatible-mode/v1",
  });
}

export async function POST(request: NextRequest) {
  try {
    let body;
    try {
      body = await request.json();
    } catch {
      return NextResponse.json(
        { error: "请求格式错误，请检查请求体" },
        { status: 400 }
      );
    }

    const { text } = body;

    if (!text || typeof text !== "string") {
      return NextResponse.json(
        { error: "请输入有效的文本内容" },
        { status: 400 }
      );
    }

    if (text.length > MAX_CHARACTERS) {
      return NextResponse.json(
        { error: `文本长度不能超过 ${MAX_CHARACTERS} 字符` },
        { status: 400 }
      );
    }

    if (!text.trim()) {
      return NextResponse.json(
        { error: "请输入内容" },
        { status: 400 }
      );
    }

    const apiKey = process.env.DASHSCOPE_API_KEY;
    if (!apiKey) {
      return NextResponse.json(
        { error: "服务器配置错误：缺少 API Key" },
        { status: 500 }
      );
    }

    const openai = getOpenAIClient();

    const prompt = `你是一位专业的风险投资分析师。请分析以下初创公司落地页文案，并生成一份结构化的评估报告。

请按照以下格式返回 JSON 数据，确保所有字段都是字符串类型（score 字段除外，它是 0-100 的整数）：

{
  "score": 85,
  "scoreReason": "评分理由，简要说明为什么给出这个分数",
  "name": "产品/公司名称",
  "coreFunction": "核心功能描述",
  "productSellingPoint": "产品卖点",
  "applicationField": "应用领域",
  "marketAssessment": "市场评估",
  "risksAndConcerns": "风险与关注点",
  "nextSteps": "后续建议"
}

落地页文案：
${text}

请直接返回 JSON 对象，不要包含任何其他文字说明。`;

    const completion = await openai.chat.completions.create({
      model: "qwen3-max",
      messages: [
        {
          role: "system",
          content:
            "你是一位专业的风险投资分析师，擅长从落地页文案中提取关键信息并生成结构化报告。请始终以 JSON 格式返回结果。",
        },
        { role: "user", content: prompt },
      ],
      temperature: 0.7,
      max_tokens: 2000,
    });

    const content = completion.choices[0]?.message?.content;
    if (!content) {
      return NextResponse.json(
        { error: "AI 服务返回空响应" },
        { status: 500 }
      );
    }

    let report: AnalysisReport;
    try {
      const jsonMatch = content.match(/\{[\s\S]*\}/);
      if (!jsonMatch) {
        throw new Error("未找到 JSON 数据");
      }
      report = JSON.parse(jsonMatch[0]);
    } catch (parseError) {
      console.error("JSON 解析错误:", parseError);
      console.error("原始响应:", content);
      return NextResponse.json(
        { error: "AI 响应格式错误，请稍后重试" },
        { status: 500 }
      );
    }

    if (
      typeof report.score !== "number" ||
      !report.scoreReason ||
      !report.name ||
      !report.coreFunction ||
      !report.productSellingPoint ||
      !report.applicationField ||
      !report.marketAssessment ||
      !report.risksAndConcerns ||
      !report.nextSteps
    ) {
      return NextResponse.json(
        { error: "AI 返回的数据不完整，请稍后重试" },
        { status: 500 }
      );
    }

    return NextResponse.json(report);
  } catch (error) {
    console.error("API 错误:", error);
    if (error instanceof Error) {
      if (error.message.includes("API key")) {
        return NextResponse.json(
          { error: "API Key 配置错误" },
          { status: 500 }
        );
      }
    }
    return NextResponse.json(
      { error: "分析失败，请稍后重试" },
      { status: 500 }
    );
  }
}

