import { POST } from "@/app/api/report/route";
import { NextRequest } from "next/server";
import OpenAI from "openai";

jest.mock("openai");

describe("/api/report", () => {
  const mockOpenAI = OpenAI as jest.MockedClass<typeof OpenAI>;
  const mockChatCompletionsCreate = jest.fn();

  beforeEach(() => {
    jest.clearAllMocks();
    process.env.DASHSCOPE_API_KEY = "test-api-key";
    
    mockOpenAI.mockImplementation(() => ({
      chat: {
        completions: {
          create: mockChatCompletionsCreate,
        },
      },
    } as any));
  });

  afterEach(() => {
    delete process.env.DASHSCOPE_API_KEY;
  });

  it("should return 400 if text is missing", async () => {
    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({}),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(400);
    expect(data.error).toBe("请输入有效的文本内容");
  });

  it("should return 400 if text exceeds MAX_CHARACTERS", async () => {
    const longText = "a".repeat(8001);
    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({ text: longText }),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(400);
    expect(data.error).toContain("文本长度不能超过");
  });

  it("should return 400 if text is empty", async () => {
    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({ text: "   " }),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(400);
    expect(data.error).toBe("请输入内容");
  });

  it("should return 500 if API key is missing", async () => {
    delete process.env.DASHSCOPE_API_KEY;

    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({ text: "测试文案" }),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(500);
    expect(data.error).toBe("服务器配置错误：缺少 API Key");
  });

  it("should return analysis report on success", async () => {
    const mockReport = {
      score: 85,
      scoreReason: "测试评分理由",
      name: "测试产品",
      coreFunction: "测试核心功能",
      productSellingPoint: "测试卖点",
      applicationField: "测试应用领域",
      marketAssessment: "测试市场评估",
      risksAndConcerns: "测试风险",
      nextSteps: "测试建议",
    };

    mockChatCompletionsCreate.mockResolvedValue({
      choices: [
        {
          message: {
            content: JSON.stringify(mockReport),
          },
        },
      ],
    });

    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({ text: "测试落地页文案" }),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(200);
    expect(data).toEqual(mockReport);
    expect(mockChatCompletionsCreate).toHaveBeenCalledTimes(1);
  });

  it("should handle JSON parsing errors", async () => {
    mockChatCompletionsCreate.mockResolvedValue({
      choices: [
        {
          message: {
            content: "这不是有效的JSON",
          },
        },
      ],
    });

    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({ text: "测试文案" }),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(500);
    expect(data.error).toBe("AI 响应格式错误，请稍后重试");
  });

  it("should handle incomplete report data", async () => {
    const incompleteReport = {
      score: 85,
      name: "测试产品",
    };

    mockChatCompletionsCreate.mockResolvedValue({
      choices: [
        {
          message: {
            content: JSON.stringify(incompleteReport),
          },
        },
      ],
    });

    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({ text: "测试文案" }),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(500);
    expect(data.error).toBe("AI 返回的数据不完整，请稍后重试");
  });

  it("should handle OpenAI API errors", async () => {
    mockChatCompletionsCreate.mockRejectedValue(
      new Error("API key is invalid")
    );

    const request = new NextRequest("http://localhost:3000/api/report", {
      method: "POST",
      body: JSON.stringify({ text: "测试文案" }),
    });

    const response = await POST(request);
    const data = await response.json();

    expect(response.status).toBe(500);
    expect(data.error).toBe("API Key 配置错误");
  });
});

