#!/usr/bin/env node
// Simple migration runner using postgres (node-postgres-js) client.
// Applies SQL files in db/migrations in lexical order and records them in __migrations.

import fs from 'fs/promises';
import path from 'path';
import crypto from 'crypto';
import postgres from 'postgres';

const MIGRATIONS_DIR = path.resolve(process.cwd(), 'db', 'migrations');

function getenv(key, fallback) {
  return process.env[key] ?? fallback;
}

// Load .env if available (very lightweight)
try {
  const envPath = path.resolve(process.cwd(), '.env');
  const raw = await fs.readFile(envPath, 'utf8');
  for (const line of raw.split(/\r?\n/)) {
    const m = line.match(/^([A-Z0-9_]+)=(.*)$/);
    if (m) {
      const [, k, v] = m;
      if (!process.env[k]) process.env[k] = v;
    }
  }
} catch {}

const baseUrl = getenv('POSTGRES_URL', null);
if (!baseUrl) {
  console.error('ERROR: POSTGRES_URL is not set in environment');
  process.exit(1);
}

const url = baseUrl.includes('sslmode=') ? baseUrl : `${baseUrl}?sslmode=require`;
const sql = postgres(url, { max: 1 });

async function ensureMigrationsTable() {
  await sql`CREATE TABLE IF NOT EXISTS public.__migrations (
    id SERIAL PRIMARY KEY,
    filename TEXT NOT NULL UNIQUE,
    checksum TEXT NOT NULL,
    applied_at TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now()
  );`;
}

async function getApplied() {
  const rows = await sql`SELECT filename, checksum FROM public.__migrations ORDER BY filename`;
  const map = new Map();
  for (const r of rows) map.set(r.filename, r.checksum);
  return map;
}

async function listMigrationFiles() {
  const entries = await fs.readdir(MIGRATIONS_DIR);
  return entries
    .filter((f) => f.endsWith('.sql'))
    .sort((a, b) => a.localeCompare(b));
}

function sha256(content) {
  return crypto.createHash('sha256').update(content).digest('hex');
}

async function applyMigration(filename, content) {
  console.log(`\nApplying migration: ${filename}`);
  await sql.begin(async (trx) => {
    await trx.unsafe(content);
    await trx`INSERT INTO public.__migrations (filename, checksum) VALUES (${filename}, ${sha256(content)})`;
  });
  console.log(`✓ Applied ${filename}`);
}

async function main() {
  await ensureMigrationsTable();
  const applied = await getApplied();
  const files = await listMigrationFiles();
  if (files.length === 0) {
    console.log('No migration files found');
    await sql.end();
    return;
  }
  for (const f of files) {
    const full = path.join(MIGRATIONS_DIR, f);
    const content = await fs.readFile(full, 'utf8');
    const sum = sha256(content);
    if (applied.has(f)) {
      const prev = applied.get(f);
      if (prev !== sum) {
        console.error(`ERROR: checksum mismatch for ${f}. Previously applied differs.`);
        process.exit(1);
      }
      console.log(`- Skipping already applied: ${f}`);
      continue;
    }
    await applyMigration(f, content);
  }
  await sql.end();
  console.log('\nAll migrations up to date.');
}

main().catch(async (err) => {
  console.error(err);
  try { await sql.end(); } catch {}
  process.exit(1);
});

