# H5 英语单词学习 - 技术方案（Tech Spec）

版本：v0.1  
更新时间：2025-09-08  
对应 PRD：`docs/prd.md`

## 1. 技术栈与总体架构
- 前端：Next.js 14（App Router, RSC + Server Actions）、React 18、Tailwind CSS。
- 认证：NextAuth v5（Credentials）。当前代码使用自建 `User` 表（见 `app/db.ts`），后续可接入 Drizzle Adapter 以持久化 session/token（可选）。
- 数据层：PostgreSQL（`postgres` 驱动 + `drizzle-orm`）。
- 部署：Vercel（前端）+ 远程 Postgres（Supabase/Neon/Vercel Postgres 等）。
- 运行环境：Node.js ≥ 18。

架构要点：
- 读多写少：词书/单词数据以 `books`/`words` 表为主；学习过程以“进度 + 事件日志 + 单词状态”三类表实现。
- RSC 拉取展示数据，Server Actions 执行写操作（如推进进度）。
- 路由：`/` 首页、`/me` 我的、`/learn/[bookId]` 学习页、`/word/[bookId]/[idx]` 详情页。

## 2. 现有数据表（已提供）
```sql
create table public.words (
  id bigint generated by default as identity not null,
  "wordRank" integer null,
  "headWord" text null,
  content json null,
  "bookId" text null,
  constraint words_pkey primary key (id)
);
```

```sql
create table public.books (
  id serial not null,
  title text not null,
  word_count integer not null,
  cover_url text null,
  book_id text not null,
  tags text null,
  created_at timestamp without time zone not null default now(),
  constraint books_pkey primary key (id),
  constraint books_book_id_unique unique (book_id)
);
```

说明：
- `words.content` 存储完整 JSON（建议未来改为 `jsonb` 以利索引，但本期兼容现状）。
- `books.book_id` 为业务上的外键键值；与 `words.bookId` 对应。

## 3. 新增数据模型设计（依据 PRD）
为支持“最近学习”“断点续学”“逐词推进”“详情与收藏/难度”等，新增以下表：

### 3.1 用户表（沿用现状）
当前通过代码动态创建：
```sql
create table "User" (
  id serial primary key,
  email varchar(64),
  password varchar(64)
);
```
建议补充：`constraint user_email_unique unique (email)`（防止重复注册）。

### 3.2 词书进度表 `user_book_progress`
用于记录每位用户在每本词书的学习进度（断点续学与“最近学习”数据源）。
```sql
create table public.user_book_progress (
  id bigserial primary key,
  user_id integer not null references "User"(id) on delete cascade,
  book_id text not null references public.books(book_id) on delete cascade,
  last_idx integer not null default -1, -- -1 表示未开始，进入学习从 0 开始
  last_word_id bigint null references public.words(id) on delete set null,
  updated_at timestamp without time zone not null default now(),
  created_at timestamp without time zone not null default now(),
  constraint ubp_user_book_unique unique (user_id, book_id)
);
create index idx_ubp_user_updated on public.user_book_progress(user_id, updated_at desc);
```

业务规则：
- 进入 `/learn/[bookId]`：
  - 无记录→创建 `{ last_idx: -1 }`，起始 `nextIdx = 0`。
  - 有记录→从 `nextIdx = last_idx + 1` 开始，越界则视为完成。
- 点击“下一条”→`last_idx = currentIdx`、`last_word_id = currentWordId`、`updated_at = now()`。

### 3.3 单词学习状态表 `user_word_state`
用于记录某用户对某个单词的个性化状态：是否收藏、难度、复习节奏等（为将来 SRS/错题本扩展预留）。
```sql
-- 可选：枚举类型
-- create type word_difficulty as enum ('unknown', 'learning', 'hard', 'known');

create table public.user_word_state (
  id bigserial primary key,
  user_id integer not null references "User"(id) on delete cascade,
  word_id bigint not null references public.words(id) on delete cascade,
  book_id text not null references public.books(book_id) on delete cascade,
  starred boolean not null default false,      -- 收藏/标记
  difficulty text not null default 'learning', -- 可用 check 约束代替 enum
  exposures integer not null default 0,        -- 接触次数
  last_seen_at timestamp without time zone null,
  next_review_at timestamp without time zone null, -- 预留给 SRS
  ef real null,         -- 记忆强度（SRS 预留）
  interval integer null,-- 天数间隔（SRS 预留）
  repetition integer null,
  created_at timestamp without time zone not null default now(),
  updated_at timestamp without time zone not null default now(),
  constraint uws_user_word_unique unique (user_id, word_id),
  constraint uws_difficulty_check check (difficulty in ('unknown','learning','hard','known'))
);
create index idx_uws_user_book on public.user_word_state(user_id, book_id);
create index idx_uws_user_next_review on public.user_word_state(user_id, next_review_at);
```

业务规则：
- 学习页展示时可把 `exposures+1`，`last_seen_at=now()`。
- 点击“收藏”→`starred=true`；点击“难”→`difficulty='hard'`。
- 未来启用 SRS 时更新 `ef/interval/repetition/next_review_at`。

### 3.4 学习事件日志 `study_event`
用于简单行为分析与问题排查（可异步入库）。
```sql
-- create type study_action as enum (
--   'view','next','open_detail','mark_known','mark_hard','star','unstar'
-- );

create table public.study_event (
  id bigserial primary key,
  user_id integer not null references "User"(id) on delete cascade,
  book_id text not null references public.books(book_id) on delete cascade,
  word_id bigint null references public.words(id) on delete set null,
  action text not null, -- 用 check 约束代替 enum，便于迁移
  meta jsonb null,      -- 额外上下文（设备、耗时、页面来源等）
  created_at timestamp without time zone not null default now(),
  constraint se_action_check check (action in (
    'view','next','open_detail','mark_known','mark_hard','star','unstar'
  ))
);
create index idx_se_user_time on public.study_event(user_id, created_at desc);
```

备注：若日志量较大，可转冷存储或仅保留近 90 天。

## 4. 数据访问层与 Drizzle Schema 建议
为便于类型安全，建议将新表声明为 Drizzle Schema；示例（TypeScript 片段）：
```ts
// db/schema.ts
import { pgTable, serial, integer, bigint, text, boolean, timestamp, jsonb, primaryKey } from 'drizzle-orm/pg-core';

export const userBookProgress = pgTable('user_book_progress', {
  id: bigint('id', { mode: 'number' }).primaryKey(),
  userId: integer('user_id').notNull(),
  bookId: text('book_id').notNull(),
  lastIdx: integer('last_idx').notNull().default(-1),
  lastWordId: bigint('last_word_id', { mode: 'number' }),
  updatedAt: timestamp('updated_at', { withTimezone: false }).notNull().defaultNow(),
  createdAt: timestamp('created_at', { withTimezone: false }).notNull().defaultNow(),
}, (t) => ({
  userBookUnique: primaryKey({ columns: [t.userId, t.bookId], name: 'ubp_user_book_unique' }),
}));

export const userWordState = pgTable('user_word_state', {
  id: bigint('id', { mode: 'number' }).primaryKey(),
  userId: integer('user_id').notNull(),
  wordId: bigint('word_id', { mode: 'number' }).notNull(),
  bookId: text('book_id').notNull(),
  starred: boolean('starred').notNull().default(false),
  difficulty: text('difficulty').notNull().default('learning'),
  exposures: integer('exposures').notNull().default(0),
  lastSeenAt: timestamp('last_seen_at'),
  nextReviewAt: timestamp('next_review_at'),
  ef: text('ef'), // 也可用 numeric/real
  interval: integer('interval'),
  repetition: integer('repetition'),
  createdAt: timestamp('created_at').notNull().defaultNow(),
  updatedAt: timestamp('updated_at').notNull().defaultNow(),
}, (t) => ({
  userWordUnique: primaryKey({ columns: [t.userId, t.wordId], name: 'uws_user_word_unique' }),
}));

export const studyEvent = pgTable('study_event', {
  id: bigint('id', { mode: 'number' }).primaryKey(),
  userId: integer('user_id').notNull(),
  bookId: text('book_id').notNull(),
  wordId: bigint('word_id', { mode: 'number' }),
  action: text('action').notNull(),
  meta: jsonb('meta'),
  createdAt: timestamp('created_at').notNull().defaultNow(),
});
```

注意：若继续使用现有 `app/db.ts` 动态建表，新增表建议通过独立迁移 SQL 执行，以减少运行时副作用。

## 5. 后端接口与 Server Actions 设计
与 PRD 对应的能力：

- 书籍与单词
  - 获取词书列表（RSC）
    - 输入：无
    - 输出：`[{ bookId, title, wordCount, coverUrl, tags }]`
  - 获取词书详细/分页单词（RSC/Action）
    - 输入：`bookId, offset, limit`
    - 输出：`[{ id, idx, headWord, phoneticUk/Us, transCn, exampleEn, ... }]`

- 进度与最近学习
  - 获取用户最近学习（RSC）
    - 输入：`userId`
    - 输出：最近 3 条 `user_book_progress` 按 `updated_at desc`
  - 读取/初始化进度（Action）
    - 输入：`userId, bookId`
    - 逻辑：无则 insert `{last_idx:-1}` 并返回 `nextIdx=0`；否则返回 `nextIdx=last_idx+1`
  - 推进进度（Action）
    - 输入：`userId, bookId, currentIdx, currentWordId`
    - 逻辑：`last_idx=currentIdx, last_word_id=currentWordId, updated_at=now()`

- 单词状态
  - 标记收藏（Action）：`userId, wordId, bookId, starred=true/false`
  - 标记难度（Action）：`difficulty in ('unknown','learning','hard','known')`
  - 记录曝光（Action）：`exposures+=1, last_seen_at=now()`

- 事件日志
  - 记录学习事件（Action）（可选/异步）：`action in ('view','next','open_detail',...)`

接口形态建议：
- 尽量使用 Server Actions 直接操作 DB（减少 API 层样板），读取用 RSC 在服务端执行。
- 若需要客户端缓存/离线，可增加 `/api/*` route 的 REST 端点。

## 6. 关键查询与数据映射
- 从 `words.content` 中解析字段：
  - `phoneticUk` ← `content->'word'->'content'->>'ukphone'`
  - `phoneticUs` ← `content->'word'->'content'->>'usphone'`
  - `transCn` ← `content->'word'->'content'->'trans'->0->>'tranCn'`
  - `exampleEn` ← `content->'word'->'content'->'sentence'->'sentences'->0->>'sContent'`
  - `exampleCn` ← `...->>'sCn'`

样例：
```sql
select id,
       "headWord",
       content->'word'->'content'->>'ukphone'  as phonetic_uk,
       content->'word'->'content'->>'usphone'  as phonetic_us,
       content->'word'->'content'->'trans'->0->>'tranCn' as trans_cn,
       content->'word'->'content'->'sentence'->'sentences'->0->>'sContent' as example_en
from public.words
where "bookId" = $1 and "wordRank" is not null
order by "wordRank"
limit $2 offset $3;
```

说明：若 `words` 的顺序以 `idx` 为准，可在导入时把 `idx` 存入 `wordRank` 或新增列；MVP 按 `wordRank` 排序兼容。

## 7. 前端实现要点
- 组件：
  - 底部 TabBar（`HomeTab`/`MeTab`）固定在布局组件中。
  - 首页：
    - 已登录：渲染“最近学习”+ 词书列表。
    - 未登录：仅词书列表；点击词书→`router.push('/me?auth=login')`。
  - 我的页：显示邮箱、词书进度、退出登录（`signOut()`）。未登录时打开登录弹窗。
  - 学习页：
    - 首次加载调用“读取/初始化进度”Action，取 `nextIdx` 与单词数据。
    - 展示 `WordCard`（`headWord + phonetic + 简释 + 示例`）。
    - “下一条”触发推进进度 + 拉取下一词；同时更新 `user_word_state.exposures`。
    - 点击单词→跳转详情页。
  - 详情页：渲染 JSON 中的英释/同近/记忆等；提供“继续学习”。

- 数据提取：
  - RSC（服务端）直接查库并返回序列化数据。
  - 写操作使用 Server Actions，失败时在客户端提示重试。

- 状态管理：
  - 以 URL 为状态（`bookId/idx`），进度交给后端持久化；无需在客户端冗余存储。

## 8. 权限与安全
- 所有 Server Actions 在服务端校验 `auth()`，仅允许本人读写 `userId` 关联数据。
- 输入校验：采用 `zod`/内置校验，对 `bookId`/`idx`/`wordId` 做类型与范围校验。
- 防重入：推进进度时要求 `currentIdx = last_idx + 1`（或乐观锁 compare-and-swap），避免回退/跳跃。
- 数据最小化：客户端不暴露数据库连接串与私密字段。

## 9. 性能与可用性
- 列表分页：词书与单词均分页/流式加载（尤其是单词列表）。
- RSC 缓存：词书列表可 `revalidate`（如 60s）；学习页为强动态（`cache: 'no-store'`）。
- 索引：
  - `user_book_progress(user_id, updated_at desc)` 支撑“最近学习”。
  - `user_word_state(user_id, book_id)` 支撑我的页进度统计。
- JSON 解析：在 SQL 层取所需字段，减少网络负载。

## 10. 监控与埋点（可选）
- `study_event` 作为后端事实表，前端无需额外 SDK；重要路径（next、open_detail）落库。
- 线上异常：可接入 Sentry（前端+后端）。

## 11. 迁移与上线步骤
1) 创建新表与索引：执行本文件中 SQL（`user_book_progress`、`user_word_state`、`study_event`，以及 `User.email` 唯一约束）。  
2) 接入 Drizzle Schema（或保留 SQL 方案）并实现 Server Actions。  
3) 实现前端页面骨架与调用链路；联调登录/注册逻辑。  
4) 自测关键流程与空态；预备演示数据。  
5) 上线与观察指标。

## 12. 示例 Server Actions（伪代码）
```ts
// app/actions/progress.ts
'use server';
import { auth } from 'app/auth';
import { db } from 'app/db'; // 假设封装了 drizzle client

export async function initOrGetProgress(bookId: string) {
  const session = await auth();
  if (!session?.user) throw new Error('UNAUTHORIZED');
  const userId = Number(session.user.id); // 或者通过 email 反查

  // 查询现有进度
  let row = await db.query.userBookProgress.findFirst({
    where: (t, { eq, and }) => and(eq(t.userId, userId), eq(t.bookId, bookId)),
  });
  if (!row) {
    row = await db.insert(userBookProgress).values({ userId, bookId, lastIdx: -1 }).returning().then(r => r[0]);
    return { nextIdx: 0, progress: row };
  }
  return { nextIdx: row.lastIdx + 1, progress: row };
}

export async function advanceProgress(bookId: string, currentIdx: number, currentWordId: number) {
  const session = await auth();
  if (!session?.user) throw new Error('UNAUTHORIZED');
  const userId = Number(session.user.id);

  // 乐观检查：currentIdx 应等于 last_idx + 1
  const existing = await db.query.userBookProgress.findFirst({ where: (t, { eq, and }) => and(eq(t.userId, userId), eq(t.bookId, bookId)) });
  if (!existing) throw new Error('PROGRESS_NOT_FOUND');
  if (existing.lastIdx + 1 !== currentIdx) {
    throw new Error('OUT_OF_ORDER');
  }

  await db.update(userBookProgress)
    .set({ lastIdx: currentIdx, lastWordId: currentWordId, updatedAt: new Date() })
    .where((t, { eq, and }) => and(eq(t.userId, userId), eq(t.bookId, bookId)));

  // 可选：更新曝光/事件
}
```

## 13. 进度统计与“我的”页面数据
- 每本书进度：`completed = last_idx + 1`，总词数从 `books.word_count` 获取。
- 汇总展示：查询 `user_book_progress` 与 `books` 关联，返回 `{ title, bookId, completed, total, updatedAt }`。按 `updatedAt desc` 排序。

## 14. 风险与替代方案
- `words.content` 为 `json`（非 `jsonb`）：复杂筛选/索引受限 → 短期通过应用层解析，长期建议迁移为 `jsonb` 并建立 GIN 索引。
- `User` 表动态创建：建议纳入迁移并加唯一键（email），并考虑 NextAuth Adapter 以避免自维护 session/token。
- 大表扫描：为首页/学习页设计限流与分页，避免一次性加载全书。

---

本技术文档覆盖了数据模型（含新增表）、后端 Server Actions、前端组件与数据流、权限与性能等。根据本方案实施可完整支撑 `docs/prd.md` 的 MVP 需求，并为后续 SRS/书签/报表留足扩展空间。

