-- 001_create_learning_schema.sql
-- Purpose: Create user, progress, word state, and study event tables with indexes.
-- Idempotent: safe to run multiple times.

-- 1) Ensure User table exists (align with app/db.ts)
CREATE TABLE IF NOT EXISTS public."User" (
  id SERIAL PRIMARY KEY,
  email VARCHAR(64),
  password VARCHAR(64)
);

-- 1.1) Ensure unique constraint on email
DO $$
BEGIN
  IF NOT EXISTS (
    SELECT 1 FROM pg_constraint
    WHERE conname = 'user_email_unique'
  ) THEN
    ALTER TABLE public."User"
      ADD CONSTRAINT user_email_unique UNIQUE (email);
  END IF;
END $$;

-- 2) Create user_book_progress
CREATE TABLE IF NOT EXISTS public.user_book_progress (
  id BIGSERIAL PRIMARY KEY,
  user_id INTEGER NOT NULL REFERENCES public."User"(id) ON DELETE CASCADE,
  book_id TEXT NOT NULL REFERENCES public.books(book_id) ON DELETE CASCADE,
  last_idx INTEGER NOT NULL DEFAULT -1,
  last_word_id BIGINT NULL REFERENCES public.words(id) ON DELETE SET NULL,
  updated_at TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now(),
  created_at TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now()
);

-- 2.1) Unique (user_id, book_id)
DO $$
BEGIN
  IF NOT EXISTS (
    SELECT 1 FROM pg_constraint WHERE conname = 'ubp_user_book_unique'
  ) THEN
    ALTER TABLE public.user_book_progress
      ADD CONSTRAINT ubp_user_book_unique UNIQUE (user_id, book_id);
  END IF;
END $$;

-- 2.2) Index for recent learning
CREATE INDEX IF NOT EXISTS idx_ubp_user_updated
  ON public.user_book_progress(user_id, updated_at DESC);

-- 3) Create user_word_state
CREATE TABLE IF NOT EXISTS public.user_word_state (
  id BIGSERIAL PRIMARY KEY,
  user_id INTEGER NOT NULL REFERENCES public."User"(id) ON DELETE CASCADE,
  word_id BIGINT NOT NULL REFERENCES public.words(id) ON DELETE CASCADE,
  book_id TEXT NOT NULL REFERENCES public.books(book_id) ON DELETE CASCADE,
  starred BOOLEAN NOT NULL DEFAULT FALSE,
  difficulty TEXT NOT NULL DEFAULT 'learning',
  exposures INTEGER NOT NULL DEFAULT 0,
  last_seen_at TIMESTAMP WITHOUT TIME ZONE NULL,
  next_review_at TIMESTAMP WITHOUT TIME ZONE NULL,
  ef REAL NULL,
  interval INTEGER NULL,
  repetition INTEGER NULL,
  created_at TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now(),
  updated_at TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now()
);

-- 3.1) Unique (user_id, word_id)
DO $$
BEGIN
  IF NOT EXISTS (
    SELECT 1 FROM pg_constraint WHERE conname = 'uws_user_word_unique'
  ) THEN
    ALTER TABLE public.user_word_state
      ADD CONSTRAINT uws_user_word_unique UNIQUE (user_id, word_id);
  END IF;
END $$;

-- 3.2) Difficulty check constraint
DO $$
BEGIN
  IF NOT EXISTS (
    SELECT 1 FROM pg_constraint WHERE conname = 'uws_difficulty_check'
  ) THEN
    ALTER TABLE public.user_word_state
      ADD CONSTRAINT uws_difficulty_check CHECK (
        difficulty IN ('unknown','learning','hard','known')
      );
  END IF;
END $$;

-- 3.3) Indexes
CREATE INDEX IF NOT EXISTS idx_uws_user_book
  ON public.user_word_state(user_id, book_id);
CREATE INDEX IF NOT EXISTS idx_uws_user_next_review
  ON public.user_word_state(user_id, next_review_at);

-- 4) Create study_event
CREATE TABLE IF NOT EXISTS public.study_event (
  id BIGSERIAL PRIMARY KEY,
  user_id INTEGER NOT NULL REFERENCES public."User"(id) ON DELETE CASCADE,
  book_id TEXT NOT NULL REFERENCES public.books(book_id) ON DELETE CASCADE,
  word_id BIGINT NULL REFERENCES public.words(id) ON DELETE SET NULL,
  action TEXT NOT NULL,
  meta JSONB NULL,
  created_at TIMESTAMP WITHOUT TIME ZONE NOT NULL DEFAULT now()
);

-- 4.1) action check constraint
DO $$
BEGIN
  IF NOT EXISTS (
    SELECT 1 FROM pg_constraint WHERE conname = 'se_action_check'
  ) THEN
    ALTER TABLE public.study_event
      ADD CONSTRAINT se_action_check CHECK (
        action IN ('view','next','open_detail','mark_known','mark_hard','star','unstar')
      );
  END IF;
END $$;

-- 4.2) Index for timeline
CREATE INDEX IF NOT EXISTS idx_se_user_time
  ON public.study_event(user_id, created_at DESC);

