import { auth } from '@/app/auth';
import HomePage from '@/app/components/HomePage';
import { listBooks, listRecentLearningByUser, getUser } from '@/app/db';
import type { Book } from '@/app/lib/mock-data';

export default async function Page() {
  const session = await auth();
  // Fetch books from DB and map to UI Book shape
  let books: Book[] = [];
  let recent: Array<{ id: number; book: Book; lastIdx: number; updatedAt: Date }> = [];
  try {
    const rows = await listBooks();
    books = rows.map((r) => ({
      id: r.id,
      bookId: r.book_id,
      title: r.title,
      wordCount: r.word_count,
      coverUrl: r.cover_url || undefined,
      tags: r.tags || undefined,
      createdAt: new Date(r.created_at as any),
    }));

    if (session?.user) {
      // Resolve user id
      let userId: number | null = null;
      const maybeId = (session.user as any)?.id;
      const email = (session.user as any)?.email as string | undefined;
      if (maybeId !== undefined) {
        const n = Number(maybeId);
        if (!Number.isNaN(n)) userId = n;
      }
      if (!userId && email) {
        const users = await getUser(email);
        if (users && users.length > 0) userId = Number(users[0].id);
      }

      if (userId) {
        const rec = await listRecentLearningByUser(userId, 3);
        // map join rows to UI
        recent = rec.map((r) => ({
          id: r.id,
          lastIdx: r.last_idx,
          updatedAt: new Date(r.updated_at as any),
          book: {
            id: r.b_id,
            bookId: r.book_id,
            title: r.title,
            wordCount: r.word_count,
            coverUrl: r.cover_url || undefined,
            tags: r.tags || undefined,
            createdAt: new Date(),
          },
        }));
      }
    }
  } catch (e) {
    // If DB is not reachable, fallback is handled inside HomePage if needed
  }

  return <HomePage session={session} books={books} recent={recent} />;
}
