// Mock data for H5 English Learning App

// Types based on database schema
export interface Book {
  id: number;
  bookId: string;
  title: string;
  wordCount: number;
  coverUrl?: string;
  tags?: string;
  createdAt: Date;
}

export interface Word {
  id: number;
  idx: number;
  wordRank: number;
  headWord: string;
  content: any; // JSON content
  bookId: string;
}

export interface UserBookProgress {
  id: number;
  userId: number;
  bookId: string;
  lastIdx: number;
  lastWordId?: number;
  updatedAt: Date;
  createdAt: Date;
}

export interface UserWordState {
  id: number;
  userId: number;
  wordId: number;
  bookId: string;
  starred: boolean;
  difficulty: 'unknown' | 'learning' | 'hard' | 'known';
  exposures: number;
  lastSeenAt?: Date;
  nextReviewAt?: Date;
}

// Helper function to decode hex-encoded URLs
function decodeHexUrl(hexUrl: string): string {
  // Extract the hex part from GitHub camo URL
  const hexPart = hexUrl.split('/').pop();
  if (!hexPart) return hexUrl;
  
  try {
    // Convert hex to string
    const decoded = hexPart.match(/.{1,2}/g)?.map(byte => String.fromCharCode(parseInt(byte, 16))).join('');
    return decoded || hexUrl;
  } catch {
    return hexUrl;
  }
}

// Mock Books Data - Using real data with decoded URLs
export const mockBooks: Book[] = [
  {
    id: 1,
    bookId: 'PEPXiaoXue3_1',
    title: '人教版小学英语-三年级上册',
    wordCount: 64,
    coverUrl: 'https://nos.netease.com/ydschool-online/1496632776935Level4luan_1.jpg',
    tags: '小学,三年级,人教版',
    createdAt: new Date('2025-09-08T10:22:50.873273')
  },
  {
    id: 3,
    bookId: 'TOEFL_3',
    title: '新东方 TOEFL 词汇',
    wordCount: 4264,
    coverUrl: 'https://nos.netease.com/ydschool-online/newOriental_TOEFL_3.jpg',
    tags: '托福,新东方',
    createdAt: new Date('2025-09-08T10:33:28.939207')
  }
];

// Mock Words Data - Based on the provided JSON structure
export const mockWords: Word[] = [
  {
    id: 1,
    idx: 0,
    wordRank: 1,
    headWord: 'ruler',
    bookId: 'PEPXiaoXue3_1',
    content: {
      word: {
        wordId: 'PEPXiaoXue3_1_1',
        content: {
          ukphone: 'ˈruːlə',
          usphone: 'ˈruːlər',
          trans: [
            {
              descCn: '中释',
              tranCn: '尺子',
              descOther: '英释',
              tranOther: 'a long flat piece of plastic, metal, or wood that you use for measuring things or drawing straight lines'
            }
          ],
          sentence: {
            desc: '例句',
            sentences: [
              {
                sCn: '我需要一把12英寸的尺子。',
                sContent: 'I need a 12-inch ruler.',
                sContent_eng: 'I need a 12-inch <b>ruler</b>.'
              }
            ]
          },
          syno: {
            desc: '同近',
            synos: [
              {
                pos: 'n',
                hwds: [{ w: 'straightedge' }, { w: 'measuring stick' }],
                tran: '直尺，测量尺'
              }
            ]
          },
          remMethod: {
            val: 'rule(规则) + r → 用尺子画直线就像定规则'
          }
        },
        wordHead: 'ruler'
      }
    }
  },
  {
    id: 2,
    idx: 1,
    wordRank: 2,
    headWord: 'pencil',
    bookId: 'PEPXiaoXue3_1',
    content: {
      word: {
        wordId: 'PEPXiaoXue3_1_2',
        content: {
          ukphone: 'ˈpensl',
          usphone: 'ˈpensl',
          trans: [
            {
              descCn: '中释',
              tranCn: '铅笔',
              descOther: '英释',
              tranOther: 'a narrow piece of wood or plastic with a black or colored substance through the middle, used for writing or drawing'
            }
          ],
          sentence: {
            desc: '例句',
            sentences: [
              {
                sCn: '请用铅笔写下你的名字。',
                sContent: 'Please write your name with a pencil.',
                sContent_eng: 'Please write your name with a <b>pencil</b>.'
              }
            ]
          },
          syno: {
            desc: '同近',
            synos: [
              {
                pos: 'n',
                hwds: [{ w: 'lead pencil' }, { w: 'writing implement' }],
                tran: '铅笔，书写工具'
              }
            ]
          },
          remMethod: {
            val: 'pen(笔) + cil → 像笔一样的工具'
          }
        },
        wordHead: 'pencil'
      }
    }
  },
  {
    id: 3,
    idx: 2,
    wordRank: 3,
    headWord: 'eraser',
    bookId: 'PEPXiaoXue3_1',
    content: {
      word: {
        wordId: 'PEPXiaoXue3_1_3',
        content: {
          ukphone: 'ɪˈreɪzə',
          usphone: 'ɪˈreɪsər',
          trans: [
            {
              descCn: '中释',
              tranCn: '橡皮擦',
              descOther: '英释',
              tranOther: 'a small piece of rubber that you use to remove pencil marks from paper'
            }
          ],
          sentence: {
            desc: '例句',
            sentences: [
              {
                sCn: '我需要一块橡皮擦来擦掉错误。',
                sContent: 'I need an eraser to remove the mistakes.',
                sContent_eng: 'I need an <b>eraser</b> to remove the mistakes.'
              }
            ]
          },
          syno: {
            desc: '同近',
            synos: [
              {
                pos: 'n',
                hwds: [{ w: 'rubber' }, { w: 'rubber eraser' }],
                tran: '橡皮，橡皮擦'
              }
            ]
          },
          remMethod: {
            val: 'erase(擦除) + r → 用来擦除的工具'
          }
        },
        wordHead: 'eraser'
      }
    }
  },
  {
    id: 4,
    idx: 3,
    wordRank: 4,
    headWord: 'crayon',
    bookId: 'PEPXiaoXue3_1',
    content: {
      word: {
        wordId: 'PEPXiaoXue3_1_4',
        content: {
          syno: {
            desc: '同近',
            synos: [
              {
                pos: 'n',
                hwds: [{ w: 'pastels' }],
                tran: '[轻]蜡笔，有色粉笔'
              }
            ]
          },
          trans: [
            {
              descCn: '中释',
              tranCn: '蜡笔',
              descOther: '英释',
              tranOther: 'a stick of coloured wax or chalk that children use to draw pictures'
            }
          ],
          ukphone: 'ˈkreɪən',
          usphone: 'kreən',
          sentence: {
            desc: '例句',
            sentences: [
              {
                sCn: '他用蜡笔给画上色。',
                sSpeech: 'He+coloured+the+picture+with+crayon.&le=eng',
                sContent: 'He coloured the picture with crayon.',
                sContent_eng: 'He coloured the picture with <b>crayon</b>.'
              },
              {
                sCn: '谁拿了我的蜡笔盒？',
                sSpeech: 'Who+took+my+crayon+box%3F&le=eng',
                sContent: 'Who took my crayon box?',
                sContent_eng: 'Who took my <b>crayon</b> box?'
              }
            ]
          },
          remMethod: {
            val: 'cray(像cry哭) + on → 小孩哭着要的画画工具'
          },
          ukspeech: 'crayon&type=1',
          usspeech: 'crayon&type=2'
        },
        wordHead: 'crayon'
      }
    }
  },
  {
    id: 5,
    idx: 4,
    wordRank: 5,
    headWord: 'pen',
    bookId: 'PEPXiaoXue3_1',
    content: {
      word: {
        wordId: 'PEPXiaoXue3_1_5',
        content: {
          ukphone: 'pen',
          usphone: 'pen',
          trans: [
            {
              descCn: '中释',
              tranCn: '钢笔',
              descOther: '英释',
              tranOther: 'an instrument for writing or drawing with ink'
            }
          ],
          sentence: {
            desc: '例句',
            sentences: [
              {
                sCn: '请用钢笔签名。',
                sContent: 'Please sign with a pen.',
                sContent_eng: 'Please sign with a <b>pen</b>.'
              }
            ]
          },
          syno: {
            desc: '同近',
            synos: [
              {
                pos: 'n',
                hwds: [{ w: 'fountain pen' }, { w: 'ballpoint' }],
                tran: '钢笔，圆珠笔'
              }
            ]
          },
          remMethod: {
            val: 'pen就是笔，最简单的单词之一'
          }
        },
        wordHead: 'pen'
      }
    }
  }
];

// Mock User Progress Data
export const mockUserProgress: UserBookProgress[] = [
  {
    id: 1,
    userId: 1,
    bookId: 'PEPXiaoXue3_1',
    lastIdx: 2,
    lastWordId: 3,
    updatedAt: new Date('2025-09-08T10:30:00'),
    createdAt: new Date('2025-09-01T08:00:00')
  },
  {
    id: 2,
    userId: 1,
    bookId: 'TOEFL_3',
    lastIdx: 150,
    lastWordId: 151,
    updatedAt: new Date('2025-09-07T15:20:00'),
    createdAt: new Date('2025-09-02T09:00:00')
  }
];

// Mock User Word State Data
export const mockUserWordStates: UserWordState[] = [
  {
    id: 1,
    userId: 1,
    wordId: 1,
    bookId: 'PEPXiaoXue3_1',
    starred: false,
    difficulty: 'known',
    exposures: 3,
    lastSeenAt: new Date('2025-09-08T10:25:00')
  },
  {
    id: 2,
    userId: 1,
    wordId: 2,
    bookId: 'PEPXiaoXue3_1',
    starred: true,
    difficulty: 'learning',
    exposures: 2,
    lastSeenAt: new Date('2025-09-08T10:28:00')
  },
  {
    id: 3,
    userId: 1,
    wordId: 3,
    bookId: 'PEPXiaoXue3_1',
    starred: false,
    difficulty: 'hard',
    exposures: 1,
    lastSeenAt: new Date('2025-09-08T10:30:00')
  }
];

// Helper functions
export function getBookById(bookId: string): Book | undefined {
  return mockBooks.find(book => book.bookId === bookId);
}

export function getWordsByBookId(bookId: string): Word[] {
  return mockWords.filter(word => word.bookId === bookId);
}

export function getWordByBookIdAndIdx(bookId: string, idx: number): Word | undefined {
  return mockWords.find(word => word.bookId === bookId && word.idx === idx);
}

export function getUserProgress(userId: number): UserBookProgress[] {
  return mockUserProgress
    .filter(progress => progress.userId === userId)
    .sort((a, b) => b.updatedAt.getTime() - a.updatedAt.getTime());
}

export function getRecentLearning(userId: number, limit: number = 3): UserBookProgress[] {
  return getUserProgress(userId).slice(0, limit);
}

// Parse word content helper
export function parseWordContent(word: Word) {
  const content = word.content?.word?.content || {};
  return {
    headWord: word.headWord,
    phoneticUk: content.ukphone || '',
    phoneticUs: content.usphone || '',
    transCn: content.trans?.[0]?.tranCn || '',
    transEn: content.trans?.[0]?.tranOther || '',
    exampleEn: content.sentence?.sentences?.[0]?.sContent || '',
    exampleCn: content.sentence?.sentences?.[0]?.sCn || '',
    synonyms: content.syno?.synos?.flatMap((s: any) => s.hwds?.map((h: any) => h.w)) || [],
    memo: content.remMethod?.val || ''
  };
}

// Youdao dict voice API builder
// type: 'uk' -> 1, 'us' -> 2
export function youdaoAudioUrl(word: string, type: 'uk' | 'us' = 'us') {
  const t = type === 'uk' ? 1 : 2;
  const w = (word || '').trim();
  if (!w) return '';
  return `https://dict.youdao.com/dictvoice?audio=${encodeURIComponent(w)}&type=${t}`;
}
