import { drizzle } from 'drizzle-orm/postgres-js';
import { pgTable, serial, varchar, text, integer, timestamp, bigint } from 'drizzle-orm/pg-core';
import { eq } from 'drizzle-orm';
import postgres from 'postgres';
import { genSaltSync, hashSync } from 'bcrypt-ts';

// Optionally, if not using email/pass login, you can
// use the Drizzle adapter for Auth.js / NextAuth
// https://authjs.dev/reference/adapter/drizzle
let client = postgres(`${process.env.POSTGRES_URL!}?sslmode=require`);
let db = drizzle(client);

export async function getUser(email: string) {
  const users = await ensureTableExists();
  return await db.select().from(users).where(eq(users.email, email));
}

export async function createUser(email: string, password: string) {
  const users = await ensureTableExists();
  let salt = genSaltSync(10);
  let hash = hashSync(password, salt);

  return await db.insert(users).values({ email, password: hash });
}

async function ensureTableExists() {
  const result = await client`
    SELECT EXISTS (
      SELECT FROM information_schema.tables 
      WHERE table_schema = 'public' 
      AND table_name = 'User'
    );`;

  if (!result[0].exists) {
    await client`
      CREATE TABLE "User" (
        id SERIAL PRIMARY KEY,
        email VARCHAR(64),
        password VARCHAR(64)
      );`;
  }

  const table = pgTable('User', {
    id: serial('id').primaryKey(),
    email: varchar('email', { length: 64 }),
    password: varchar('password', { length: 64 }),
  });

  return table;
}

// ===================== Books Schema & Queries =====================
export const booksTable = pgTable('books', {
  id: serial('id').primaryKey(),
  title: text('title').notNull(),
  wordCount: integer('word_count').notNull(),
  coverUrl: text('cover_url'),
  bookId: text('book_id').notNull(),
  tags: text('tags'),
  createdAt: timestamp('created_at', { withTimezone: false }).notNull().defaultNow(),
});

export type BookRow = {
  id: number;
  title: string;
  word_count: number;
  cover_url: string | null;
  book_id: string;
  tags: string | null;
  created_at: Date | string;
};

export async function listBooks(): Promise<BookRow[]> {
  // raw select to keep original column names for easy mapping
  const rows = await client<BookRow[]>`
    SELECT id, title, word_count, cover_url, book_id, tags, created_at
    FROM public.books
    ORDER BY id ASC`;
  return rows;
}

// ===================== Words Schema & Queries =====================
export const wordsTable = pgTable('words', {
  id: bigint('id', { mode: 'number' }).primaryKey(),
  wordRank: integer('wordRank'),
  headWord: text('headWord'),
  content: text('content'), // we'll fetch as JSON via raw query
  bookId: text('bookId'),
});

export type WordDbRow = {
  id: number;
  wordRank: number | null;
  headWord: string | null;
  content: any; // JSON
  bookId: string | null;
};

export async function listWordsByBook(bookId: string): Promise<WordDbRow[]> {
  const rows = await client<WordDbRow[]>`
    SELECT id, "wordRank", "headWord", content, "bookId"
    FROM public.words
    WHERE "bookId" = ${bookId}
    ORDER BY "wordRank" ASC NULLS LAST, id ASC`;
  return rows;
}

// ===================== User Book Progress =====================
export type UserBookProgressRow = {
  id: number;
  user_id: number;
  book_id: string;
  last_idx: number;
  last_word_id: number | null;
  updated_at: string;
  created_at: string;
};

export async function getOrInitUserBookProgress(userId: number, bookId: string): Promise<UserBookProgressRow> {
  const inserted = await client<UserBookProgressRow[]>`
    INSERT INTO public.user_book_progress (user_id, book_id, last_idx)
    VALUES (${userId}, ${bookId}, -1)
    ON CONFLICT (user_id, book_id) DO NOTHING
    RETURNING id, user_id, book_id, last_idx, last_word_id, updated_at, created_at`;
  if (inserted.length > 0) return inserted[0];
  const rows = await client<UserBookProgressRow[]>`
    SELECT id, user_id, book_id, last_idx, last_word_id, updated_at, created_at
    FROM public.user_book_progress
    WHERE user_id = ${userId} AND book_id = ${bookId}
    LIMIT 1`;
  return rows[0];
}

export async function getNextIndexForUser(userId: number, bookId: string): Promise<number> {
  const p = await getOrInitUserBookProgress(userId, bookId);
  return (p?.last_idx ?? -1) + 1;
}

export async function updateUserBookProgress(
  userId: number,
  bookId: string,
  lastIdx: number,
  lastWordId: number | null,
): Promise<void> {
  await client`
    UPDATE public.user_book_progress
    SET last_idx = ${lastIdx}, last_word_id = ${lastWordId}, updated_at = now()
    WHERE user_id = ${userId} AND book_id = ${bookId}`;
}

// ===================== Recent Learning =====================
export type RecentLearningRow = {
  id: number;
  user_id: number;
  book_id: string;
  last_idx: number;
  last_word_id: number | null;
  updated_at: Date;
  created_at: Date;
  // joined book fields
  b_id: number;
  title: string;
  word_count: number;
  cover_url: string | null;
  tags: string | null;
};

export async function listRecentLearningByUser(userId: number, limit = 3): Promise<RecentLearningRow[]> {
  const rows = await client<RecentLearningRow[]>`
    SELECT ubp.id,
           ubp.user_id,
           ubp.book_id,
           ubp.last_idx,
           ubp.last_word_id,
           ubp.updated_at,
           ubp.created_at,
           b.id as b_id,
           b.title,
           b.word_count,
           b.cover_url,
           b.tags
    FROM public.user_book_progress ubp
    JOIN public.books b ON b.book_id = ubp.book_id
    WHERE ubp.user_id = ${userId}
    ORDER BY ubp.updated_at DESC
    LIMIT ${limit}`;
  return rows;
}

export async function listAllLearningByUser(userId: number): Promise<RecentLearningRow[]> {
  const rows = await client<RecentLearningRow[]>`
    SELECT ubp.id,
           ubp.user_id,
           ubp.book_id,
           ubp.last_idx,
           ubp.last_word_id,
           ubp.updated_at,
           ubp.created_at,
           b.id as b_id,
           b.title,
           b.word_count,
           b.cover_url,
           b.tags
    FROM public.user_book_progress ubp
    JOIN public.books b ON b.book_id = ubp.book_id
    WHERE ubp.user_id = ${userId}
    ORDER BY ubp.updated_at DESC`;
  return rows;
}
