'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { Session } from 'next-auth';
import { 
  getBookById,
  getWordByBookIdAndIdx,
  parseWordContent,
  youdaoAudioUrl,
} from '@/app/lib/mock-data';
import { 
  ArrowLeftIcon, 
  SpeakerWaveIcon,
  StarIcon,
  BookOpenIcon,
  LightBulbIcon,
  DocumentTextIcon
} from '@heroicons/react/24/outline';
import { StarIcon as StarIconSolid } from '@heroicons/react/24/solid';

interface WordDetailPageProps {
  bookId: string;
  idx: number;
  session: Session;
  serverWord?: any | null;
}

export default function WordDetailPage({ bookId, idx, session, serverWord }: WordDetailPageProps) {
  const router = useRouter();
  const [word, setWord] = useState<any>(serverWord ?? null);
  const [starred, setStarred] = useState(false);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (serverWord) {
      setWord(serverWord);
      setLoading(false);
      return;
    }
    // Fallback to mock data if no serverWord provided
    const w = getWordByBookIdAndIdx(bookId, idx);
    if (!w) {
      router.push('/');
      return;
    }
    setWord(w);
    setLoading(false);
  }, [bookId, idx, router, serverWord]);

  const handleBack = () => {
    router.back();
  };

  const handleContinueLearning = () => {
    router.push(`/learn/${bookId}`);
  };

  const handleToggleStar = () => {
    setStarred(!starred);
    // In real app, would save this state
    console.log('Toggle star:', { bookId, idx, starred: !starred });
  };

  const playAudio = (type: 'uk' | 'us') => {
    if (!word) return;
    const w = parseWordContent(word).headWord;
    const url = youdaoAudioUrl(w, type);
    if (!url) return;
    const audio = new Audio(url);
    audio.play().catch(() => {});
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-gray-500">加载中...</div>
      </div>
    );
  }

  if (!word) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <p className="text-gray-500 mb-4">单词不存在</p>
          <button
            onClick={() => router.push('/')}
            className="text-blue-600 hover:text-blue-700"
          >
            返回首页
          </button>
        </div>
      </div>
    );
  }

  const parsedWord = parseWordContent(word);
  const raw = (word?.content?.word?.content) || {};
  const translations = Array.isArray(raw.trans) ? raw.trans : [];
  const sentences = raw.sentence?.sentences && Array.isArray(raw.sentence.sentences) ? raw.sentence.sentences : [];
  const synoGroups = raw.syno?.synos && Array.isArray(raw.syno.synos) ? raw.syno.synos : [];
  const relWordGroups = raw.relWord?.rels && Array.isArray(raw.relWord.rels) ? raw.relWord.rels : [];

  return (
    <div className="min-h-screen bg-gray-50 flex flex-col">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-10">
        <div className="px-4 py-3 flex items-center justify-between">
          <div className="flex items-center gap-3">
            <button
              onClick={handleBack}
              className="p-2 hover:bg-gray-100 rounded-lg"
            >
              <ArrowLeftIcon className="w-5 h-5 text-gray-600" />
            </button>
            <h1 className="text-lg font-semibold text-gray-900">单词详情</h1>
          </div>
          <button
            onClick={handleToggleStar}
            className="p-2 hover:bg-gray-100 rounded-lg"
          >
            {starred ? (
              <StarIconSolid className="w-5 h-5 text-yellow-500" />
            ) : (
              <StarIcon className="w-5 h-5 text-gray-600" />
            )}
          </button>
        </div>
      </header>

      {/* Content */}
      <div className="flex-1 px-4 py-6 pb-24">
        {/* Word and Phonetics */}
        <section className="bg-white rounded-lg p-6 mb-4 shadow-sm">
          <h2 className="text-3xl font-bold text-gray-900 text-center mb-4">
            {parsedWord.headWord}
          </h2>
          
          <div className="flex flex-col items-center justify-center gap-4 mb-4">
            {parsedWord.phoneticUk && (
              <div className="flex items-center gap-1">
                <span className="text-sm text-gray-500">英</span>
                <span className="text-gray-700">/{parsedWord.phoneticUk}/</span>
                <button
                  onClick={() => playAudio('uk')}
                  className="p-1 hover:bg-gray-100 rounded"
                >
                  <SpeakerWaveIcon className="w-4 h-4 text-gray-600" />
                </button>
              </div>
            )}
            {parsedWord.phoneticUs && (
              <div className="flex items-center gap-1">
                <span className="text-sm text-gray-500">美</span>
                <span className="text-gray-700">/{parsedWord.phoneticUs}/</span>
                <button
                  onClick={() => playAudio('us')}
                  className="p-1 hover:bg-gray-100 rounded"
                >
                  <SpeakerWaveIcon className="w-4 h-4 text-gray-600" />
                </button>
              </div>
            )}
          </div>
        </section>

        {/* Translations */}
        <section className="bg-white rounded-lg p-6 mb-4 shadow-sm">
          <h3 className="text-lg font-semibold text-gray-900 mb-3 flex items-center gap-2">
            <BookOpenIcon className="w-5 h-5" />
            释义
          </h3>
          {translations.length > 0 ? (
            <ul className="space-y-2">
              {translations.map((t: any, i: number) => (
                <li key={i} className="text-sm">
                  <div className="text-gray-900">{t.tranCn}</div>
                  {t.tranOther && <div className="text-gray-600">{t.tranOther}</div>}
                </li>
              ))}
            </ul>
          ) : (
            <div className="text-sm text-gray-500">暂无释义</div>
          )}
        </section>

        {/* Synonyms */}
        {Array.isArray(synoGroups) && synoGroups.length > 0 && (
          <section className="bg-white rounded-lg p-6 mb-4 shadow-sm">
            <h3 className="text-lg font-semibold text-gray-900 mb-3 flex items-center gap-2">
              <DocumentTextIcon className="w-5 h-5" />
              同近义词
            </h3>
            <ul className="space-y-2">
              {synoGroups.map((g: any, i: number) => (
                <li key={i} className="text-sm">
                  <span className="text-gray-500 mr-2">[{g.pos}]</span>
                  {Array.isArray(g.hwds) && g.hwds.length > 0 && (
                    <span className="text-gray-800 mr-2">{g.hwds.map((h: any) => h.w).filter(Boolean).join(', ')}</span>
                  )}
                  {g.tran && <span className="text-gray-500">— {g.tran}</span>}
                </li>
              ))}
            </ul>
          </section>
        )}

        {/* Examples */}
        {Array.isArray(sentences) && sentences.length > 0 && (
          <section className="bg-white rounded-lg p-6 mb-4 shadow-sm">
            <h3 className="text-lg font-semibold text-gray-900 mb-3 flex items-center gap-2">
              <DocumentTextIcon className="w-5 h-5" />
              例句
            </h3>
            <ul className="space-y-2">
              {sentences.map((s: any, i: number) => (
                <li key={i} className="text-sm">
                  <div className="text-gray-900">{s.sContent || s.sContent_eng?.replace(/<[^>]+>/g, '')}</div>
                  {s.sCn && <div className="text-gray-600">{s.sCn}</div>}
                </li>
              ))}
            </ul>
          </section>
        )}

        {/* Memory Method */}
        {raw.remMethod?.val && (
          <section className="bg-white rounded-lg p-6 mb-4 shadow-sm">
            <h3 className="text-lg font-semibold text-gray-900 mb-3 flex items-center gap-2">
              <LightBulbIcon className="w-5 h-5" />
              记忆方法
            </h3>
            <p className="text-gray-700">{raw.remMethod.val}</p>
          </section>
        )}

        {/* Related Words (RelWord) */}
        {Array.isArray(relWordGroups) && relWordGroups.length > 0 && (
          <section className="bg-white rounded-lg p-6 mb-4 shadow-sm">
            <h3 className="text-lg font-semibold text-gray-900 mb-3 flex items-center gap-2">
              <DocumentTextIcon className="w-5 h-5" />
              同根词
            </h3>
            <ul className="space-y-2">
              {relWordGroups.map((g: any, i: number) => (
                <li key={i} className="text-sm">
                  <div className="text-gray-800">
                    <span className="text-gray-500 mr-2">[{g.pos}]</span>
                    {Array.isArray(g.words) && g.words.length > 0 && (
                      <span>{g.words.map((w: any) => `${w.hwd}${w.tran ? `：${w.tran}` : ''}`).join('， ')}</span>
                    )}
                  </div>
                </li>
              ))}
            </ul>
          </section>
        )}
      </div>

      {/* Bottom Action */}
      <div className="fixed bottom-0 left-0 right-0 bg-white border-t px-4 py-4">
        <button
          onClick={handleContinueLearning}
          className="w-full bg-blue-500 hover:bg-blue-600 text-white font-medium py-3 px-6 rounded-lg transition-colors"
        >
          继续学习
        </button>
      </div>
    </div>
  );
}
