'use client';

import { Word, parseWordContent, youdaoAudioUrl } from '@/app/lib/mock-data';
import { SpeakerWaveIcon } from '@heroicons/react/24/outline';

interface WordCardProps {
  word: Word;
  onNext: () => void;
  onWordClick: () => void;
  currentIdx: number;
  totalWords: number;
}

export default function WordCard({ word, onNext, onWordClick, currentIdx, totalWords }: WordCardProps) {
  const parsedWord = parseWordContent(word);
  
  const playAudio = (e: React.MouseEvent, type: 'uk' | 'us') => {
    e.stopPropagation();
    const url = youdaoAudioUrl(parsedWord.headWord, type);
    if (!url) return;
    const audio = new Audio(url);
    audio.play().catch(() => {
      // ignore play interruption errors
    });
  };

  return (
    <div className="flex flex-col h-full bg-gray-50">
      {/* Progress Bar */}
      <div className="px-4 py-3 bg-white shadow-sm">
        <div className="flex justify-between items-center mb-2">
          <span className="text-sm text-gray-600">进度 {currentIdx + 1}/{totalWords}</span>
          <span className="text-sm text-gray-500">{Math.round(((currentIdx + 1) / totalWords) * 100)}%</span>
        </div>
        <div className="w-full bg-gray-200 rounded-full h-2">
          <div
            className="bg-blue-500 h-2 rounded-full transition-all"
            style={{ width: `${((currentIdx + 1) / totalWords) * 100}%` }}
          />
        </div>
      </div>

      {/* Word Card */}
      <div className="flex-1 flex items-center justify-center px-6 py-8 overflow-y-auto">
        <div
          onClick={onWordClick}
          className="w-full max-w-md bg-white rounded-2xl shadow-xl p-8 cursor-pointer hover:shadow-2xl transition-all transform hover:scale-[1.02]"
        >
          {/* Word */}
          <div className="text-center mb-8">
            <h1 className="text-5xl font-bold text-gray-900 mb-6 tracking-wider">
              {parsedWord.headWord}
            </h1>
            
            {/* Phonetics */}
            <div className="flex items-center justify-center gap-6">
              {parsedWord.phoneticUk && (
                <div className="flex items-center gap-2 bg-gray-50 px-3 py-2 rounded-lg">
                  <span className="text-sm font-medium text-gray-500">英</span>
                  <span className="text-base text-gray-700 font-medium">/{parsedWord.phoneticUk}/</span>
                  <button
                    onClick={(e) => playAudio(e, 'uk')}
                    className="p-1.5 hover:bg-gray-200 rounded-md transition-colors"
                  >
                    <SpeakerWaveIcon className="w-5 h-5 text-gray-600" />
                  </button>
                </div>
              )}
              {parsedWord.phoneticUs && (
                <div className="flex items-center gap-2 bg-gray-50 px-3 py-2 rounded-lg">
                  <span className="text-sm font-medium text-gray-500">美</span>
                  <span className="text-base text-gray-700 font-medium">/{parsedWord.phoneticUs}/</span>
                  <button
                    onClick={(e) => playAudio(e, 'us')}
                    className="p-1.5 hover:bg-gray-200 rounded-md transition-colors"
                  >
                    <SpeakerWaveIcon className="w-5 h-5 text-gray-600" />
                  </button>
                </div>
              )}
            </div>
          </div>

          {/* Translation */}
          <div className="mb-8">
            <p className="text-2xl font-medium text-gray-800 text-center">
              {parsedWord.transCn}
            </p>
          </div>

          {/* Example */}
          {parsedWord.exampleEn && (
            <div className="border-t pt-6 mt-6">
              <p className="text-base text-gray-700 mb-2 leading-relaxed">{parsedWord.exampleEn}</p>
              {parsedWord.exampleCn && (
                <p className="text-sm text-gray-500 leading-relaxed">{parsedWord.exampleCn}</p>
              )}
            </div>
          )}
        </div>
      </div>

      {/* Next Button */}
      <div className="px-6 pb-8 pt-4 bg-gray-50">
        <button
          onClick={onNext}
          className="w-full bg-gradient-to-r from-blue-500 to-blue-600 hover:from-blue-600 hover:to-blue-700 text-white font-semibold py-4 px-6 rounded-xl transition-all transform hover:scale-[1.02] active:scale-[0.98] flex items-center justify-center gap-3 shadow-lg"
        >
          <span className="text-lg">下一个</span>
          <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2.5} d="M13 7l5 5m0 0l-5 5m5-5H6" />
          </svg>
        </button>
      </div>
    </div>
  );
}
