'use client';

import { useState } from 'react';
import { Book, UserBookProgress } from '@/app/lib/mock-data';
import { PlayIcon, ClockIcon, BookOpenIcon } from '@heroicons/react/24/solid';
import Image from 'next/image';

interface RecentBookItemProps {
  book: Book;
  progress: UserBookProgress;
  onClick: () => void;
}

export default function RecentBookItem({ book, progress, onClick }: RecentBookItemProps) {
  const [imageError, setImageError] = useState(false);
  
  const formatDate = (date: Date) => {
    const now = new Date();
    const diff = now.getTime() - date.getTime();
    const hours = Math.floor(diff / (1000 * 60 * 60));
    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
    
    if (hours < 1) return '刚刚';
    if (hours < 24) return `${hours}小时前`;
    if (days < 7) return `${days}天前`;
    return date.toLocaleDateString('zh-CN');
  };

  const progressPercentage = Math.round(((progress.lastIdx + 1) / book.wordCount) * 100);

  return (
    <div
      onClick={onClick}
      className="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden cursor-pointer hover:shadow-md transition-all min-w-[160px] max-w-[200px]"
    >
      {/* Cover Image with 3:4 aspect ratio */}
      {book.coverUrl && !imageError ? (
        <div className="relative w-full" style={{ paddingBottom: '133.33%' }}>
          <Image
            src={book.coverUrl}
            alt={book.title}
            fill
            className="object-cover"
            sizes="200px"
            onError={() => setImageError(true)}
          />
          {/* Overlay with progress info */}
          <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/20 to-transparent" />
          <div className="absolute bottom-0 left-0 right-0 p-3">
            
            
            <div className="mb-2">
              <div className="flex justify-between text-xs text-white/90 mb-1">
                <span>进度 {progressPercentage}%</span>
                <span>{progress.lastIdx + 1}/{book.wordCount}</span>
              </div>
              <div className="w-full bg-white/30 rounded-full h-1.5">
                <div
                  className="bg-white h-1.5 rounded-full"
                  style={{ width: `${progressPercentage}%` }}
                />
              </div>
            </div>
            
            <button 
              className="w-full bg-white/90 hover:bg-white text-blue-600 text-xs font-medium py-1.5 px-2 rounded flex items-center justify-center gap-1 transition-colors"
              onClick={(e) => {
                e.stopPropagation();
                onClick();
              }}
            >
              <PlayIcon className="w-3 h-3" />
              继续学习
            </button>
          </div>
        </div>
      ) : (
        <div>
          <div className="relative w-full bg-gradient-to-br from-blue-100 to-indigo-100" style={{ paddingBottom: '100%' }}>
            <div className="absolute inset-0 flex items-center justify-center">
              <BookOpenIcon className="w-12 h-12 text-blue-300" />
            </div>
          </div>
          <div className="p-3">
            <h4 className="font-medium text-gray-900 text-sm line-clamp-2 mb-1">{book.title}</h4>
            <div className="flex items-center gap-1 mb-2">
              <ClockIcon className="w-3 h-3 text-gray-400" />
              <p className="text-xs text-gray-500">{formatDate(progress.updatedAt)}</p>
            </div>
            
            <div className="mb-2">
              <div className="flex justify-between text-xs text-gray-600 mb-1">
                <span>进度 {progressPercentage}%</span>
                <span>{progress.lastIdx + 1}/{book.wordCount}</span>
              </div>
              <div className="w-full bg-gray-200 rounded-full h-1.5">
                <div
                  className="bg-blue-500 h-1.5 rounded-full"
                  style={{ width: `${progressPercentage}%` }}
                />
              </div>
            </div>
            
            <button className="w-full bg-blue-500 hover:bg-blue-600 text-white text-xs font-medium py-1.5 px-2 rounded flex items-center justify-center gap-1 transition-colors">
              <PlayIcon className="w-3 h-3" />
              继续学习
            </button>
          </div>
        </div>
      )}
    </div>
  );
}