'use client';

import { useState, useEffect } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { Session } from 'next-auth';
import { signOut } from 'next-auth/react';
import AuthModal from '@/app/components/AuthModal';
import TabBar from '@/app/components/TabBar';
import type { Book } from '@/app/lib/mock-data';
import { 
  UserCircleIcon, 
  ArrowRightOnRectangleIcon,
  BookOpenIcon,
  CalendarIcon
} from '@heroicons/react/24/outline';

interface ProfilePageProps {
  session: Session | null;
  userProgress?: Array<{ id: number; book: Book; lastIdx: number; updatedAt: Date }>;
}

export default function ProfilePage({ session, userProgress: userProgressProp = [] }: ProfilePageProps) {
  const router = useRouter();
  const searchParams = useSearchParams();
  const [showAuthModal, setShowAuthModal] = useState(false);
  const [userProgress, setUserProgress] = useState<any[]>(userProgressProp);
  const [stats, setStats] = useState<{ learned: number; books: number }>({ learned: 0, books: 0 });

  useEffect(() => {
    // Check if auth modal should be shown
    const authParam = searchParams.get('auth');
    if (!session && (authParam === 'login' || authParam === 'register')) {
      setShowAuthModal(true);
    } else if (!session) {
      // If not logged in and no auth param, show login modal by default
      setShowAuthModal(true);
    }
  }, [session, searchParams]);

  useEffect(() => {
    if (session?.user) {
      // Normalize incoming progress array and compute percentage
      const mapped = (userProgressProp || []).map((p) => ({
        id: p.id,
        bookId: p.book.bookId,
        lastIdx: p.lastIdx,
        updatedAt: p.updatedAt,
        book: p.book,
        percentage: p.book.wordCount ? Math.round(((p.lastIdx + 1) / p.book.wordCount) * 100) : 0,
      }));
      setUserProgress(mapped);
      setStats({
        learned: mapped.reduce((sum, p) => sum + (p.lastIdx + 1), 0),
        books: mapped.length,
      });
    }
  }, [session, userProgressProp]);

  // Client-side refresh to avoid staleness
  useEffect(() => {
    let aborted = false;
    async function fetchProgress() {
      if (!session?.user) return;
      try {
        const res = await fetch('/api/me/progress', { cache: 'no-store' });
        const js = await res.json();
        if (!aborted && js?.ok) {
          const mapped = (js.data || []).map((p: any) => ({
            id: p.id,
            bookId: p.book.bookId,
            lastIdx: p.lastIdx,
            updatedAt: new Date(p.updatedAt),
            book: p.book,
            percentage: p.book.wordCount ? Math.round(((p.lastIdx + 1) / p.book.wordCount) * 100) : 0,
          }));
          setUserProgress(mapped);
          setStats({
            learned: mapped.reduce((sum: number, x: any) => sum + (x.lastIdx + 1), 0),
            books: mapped.length,
          });
        }
      } catch {}
    }
    fetchProgress();
    return () => { aborted = true; };
  }, [session]);

  const handleSignOut = async () => {
    await signOut({ callbackUrl: '/' });
  };

  const handleAuthModalClose = () => {
    setShowAuthModal(false);
    // Remove auth query param
    router.push('/me');
  };

  const formatDate = (date: Date) => {
    return new Date(date).toLocaleDateString('zh-CN', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit'
    });
  };

  if (!session) {
    return (
      <div className="min-h-screen bg-gray-50 pb-20">
        <header className="bg-white shadow-sm">
          <div className="px-4 py-4">
            <h1 className="text-xl font-bold text-gray-900">我的</h1>
          </div>
        </header>

        <div className="flex flex-col items-center justify-center py-20">
          <UserCircleIcon className="w-24 h-24 text-gray-300 mb-4" />
          <p className="text-gray-500 mb-6">请先登录以查看学习进度</p>
          <button
            onClick={() => setShowAuthModal(true)}
            className="bg-blue-500 hover:bg-blue-600 text-white font-medium py-2 px-6 rounded-lg"
          >
            登录/注册
          </button>
        </div>

        <AuthModal
          isOpen={showAuthModal}
          onClose={handleAuthModalClose}
          defaultTab={searchParams.get('auth') === 'register' ? 'register' : 'login'}
        />

        <TabBar active="me" />
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gray-50 pb-20">
      <header className="bg-white shadow-sm">
        <div className="px-4 py-4">
          <h1 className="text-xl font-bold text-gray-900">我的</h1>
        </div>
      </header>

      <div className="px-4 py-4">
        {/* User Info Section */}
        <section className="bg-white rounded-lg p-4 mb-4 shadow-sm">
          <div className="flex items-center gap-4">
            <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center">
              <UserCircleIcon className="w-10 h-10 text-blue-500" />
            </div>
            <div className="flex-1">
              <p className="text-lg font-medium text-gray-900">
                {session.user?.email || '用户'}
              </p>
              <p className="text-sm text-gray-500">邮箱账号</p>
            </div>
          </div>
        </section>

        {/* Learning Progress Section */}
        <section className="bg-white rounded-lg p-4 mb-4 shadow-sm">
          <h2 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
            <BookOpenIcon className="w-5 h-5" />
            学习进度
          </h2>
          
          {userProgress.length === 0 ? (
            <p className="text-gray-500 text-center py-4">暂无学习记录</p>
          ) : (
            <div className="space-y-3">
              {userProgress.map((progress: any) => (
                <div key={progress.id} className="border-b last:border-0 pb-3 last:pb-0">
                  <div className="flex justify-between items-start mb-2">
                    <div>
                      <h3 className="font-medium text-gray-900">{progress.book.title}</h3>
                      <div className="flex items-center gap-1 mt-1">
                        <CalendarIcon className="w-3 h-3 text-gray-400" />
                        <p className="text-xs text-gray-500">
                          上次学习：{formatDate(progress.updatedAt)}
                        </p>
                      </div>
                    </div>
              <span className="text-sm font-medium text-blue-600">{progress.percentage}%</span>
                  </div>
                  
                  <div className="flex items-center gap-2">
                    <div className="flex-1 bg-gray-200 rounded-full h-2">
                      <div
                        className="bg-blue-500 h-2 rounded-full transition-all"
                        style={{ width: `${progress.percentage}%` }}
                      />
                    </div>
                    <span className="text-xs text-gray-600">
                      {progress.lastIdx + 1}/{progress.book.wordCount}
                    </span>
                  </div>
                </div>
              ))}
            </div>
          )}
        </section>

        {/* Statistics Section */}
        <section className="bg-white rounded-lg p-4 mb-4 shadow-sm">
          <h2 className="text-lg font-semibold text-gray-900 mb-4">学习统计</h2>
          <div className="grid grid-cols-2 gap-4">
            <div className="text-center">
              <p className="text-2xl font-bold text-blue-600">{stats.learned}</p>
              <p className="text-sm text-gray-500">已学单词</p>
            </div>
            <div className="text-center">
              <p className="text-2xl font-bold text-green-600">{stats.books}</p>
              <p className="text-sm text-gray-500">学习词书</p>
            </div>
          </div>
        </section>

        {/* Sign Out Button */}
        <button
          onClick={handleSignOut}
          className="w-full bg-white hover:bg-gray-50 text-red-600 font-medium py-3 px-4 rounded-lg shadow-sm flex items-center justify-center gap-2 transition-colors"
        >
          <ArrowRightOnRectangleIcon className="w-5 h-5" />
          退出登录
        </button>
      </div>

      <TabBar active="me" />
    </div>
  );
}
