'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { Session } from 'next-auth';
import WordCard from '@/app/components/WordCard';
import { advanceProgress } from '@/app/actions/progress';
import { 
  getBookById, 
  getWordsByBookId,
  mockUserProgress 
} from '@/app/lib/mock-data';
import { ArrowLeftIcon } from '@heroicons/react/24/outline';

interface LearningPageProps {
  bookId: string;
  session: Session;
  initialIdx?: number;
  serverWords?: any[];
}

export default function LearningPage({ bookId, session, initialIdx, serverWords }: LearningPageProps) {
  const router = useRouter();
  const [currentIdx, setCurrentIdx] = useState(initialIdx ?? 0);
  const [book, setBook] = useState<any>(null);
  const [words, setWords] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [completed, setCompleted] = useState(false);

  useEffect(() => {
    // Load from server data if present
    if (serverWords && serverWords.length > 0) {
      setWords(serverWords);
      setBook({ title: getBookById(bookId)?.title || '' });
      if ((initialIdx ?? 0) >= serverWords.length) {
        setCompleted(true);
      }
      setLoading(false);
      return;
    }

    // Fallback to mock data
    const bookData = getBookById(bookId);
    if (!bookData) { router.push('/'); return; }
    setBook(bookData);
    const wordsData = getWordsByBookId(bookId);
    setWords(wordsData);
    setCurrentIdx(0);
    setLoading(false);
  }, [bookId, router, serverWords, initialIdx]);

  const handleNext = async () => {
    // Persist progress: mark currentIdx as last learned
    try {
      const currentWordId = Number(words[currentIdx]?.id ?? NaN);
      await advanceProgress(bookId, currentIdx, Number.isFinite(currentWordId) ? currentWordId : null);
    } catch (e) {
      // non-blocking persistence
      console.warn('Failed to persist progress', e);
    }

    if (currentIdx < words.length - 1) {
      setCurrentIdx((i) => i + 1);
    } else {
      setCompleted(true);
    }
  };

  const handleWordClick = () => {
    router.push(`/word/${bookId}/${currentIdx}`);
  };

  const handleBack = () => {
    router.push('/');
  };

  const handleRestart = () => {
    setCurrentIdx(0);
    setCompleted(false);
  };

  if (loading) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-gray-500">加载中...</div>
      </div>
    );
  }

  if (!book || words.length === 0) {
    return (
      <div className="min-h-screen bg-gray-50 flex items-center justify-center">
        <div className="text-center">
          <p className="text-gray-500 mb-4">词书不存在或暂无单词</p>
          <button
            onClick={handleBack}
            className="text-blue-600 hover:text-blue-700"
          >
            返回首页
          </button>
        </div>
      </div>
    );
  }

  if (completed) {
    return (
      <div className="min-h-screen bg-gray-50 flex flex-col">
        {/* Header */}
        <header className="bg-white shadow-sm">
          <div className="px-4 py-3 flex items-center gap-3">
            <button
              onClick={handleBack}
              className="p-2 hover:bg-gray-100 rounded-lg"
            >
              <ArrowLeftIcon className="w-5 h-5 text-gray-600" />
            </button>
            <h1 className="text-lg font-semibold text-gray-900">{book.title}</h1>
          </div>
        </header>

        {/* Completion Message */}
        <div className="flex-1 flex items-center justify-center px-4">
          <div className="text-center">
            <div className="w-24 h-24 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg className="w-12 h-12 text-green-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            </div>
            <h2 className="text-2xl font-bold text-gray-900 mb-2">恭喜完成！</h2>
            <p className="text-gray-600 mb-6">
              你已经完成了《{book.title}》的所有单词学习
            </p>
            <div className="space-y-3">
              <button
                onClick={handleRestart}
                className="w-full bg-blue-500 hover:bg-blue-600 text-white font-medium py-3 px-6 rounded-lg"
              >
                重新学习
              </button>
              <button
                onClick={handleBack}
                className="w-full bg-gray-100 hover:bg-gray-200 text-gray-700 font-medium py-3 px-6 rounded-lg"
              >
                返回首页
              </button>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="h-screen bg-gray-50 flex flex-col overflow-hidden">
      {/* Header */}
      <header className="bg-white shadow-sm flex-shrink-0">
        <div className="px-4 py-3 flex items-center gap-3">
          <button
            onClick={handleBack}
            className="p-2 hover:bg-gray-100 rounded-lg"
          >
            <ArrowLeftIcon className="w-5 h-5 text-gray-600" />
          </button>
          <h1 className="text-lg font-semibold text-gray-900 truncate">{book.title}</h1>
        </div>
      </header>

      {/* Word Card */}
      <div className="flex-1 min-h-0">
        <WordCard
          word={words[currentIdx]}
          onNext={handleNext}
          onWordClick={handleWordClick}
          currentIdx={currentIdx}
          totalWords={words.length}
        />
      </div>
    </div>
  );
}
