'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { Session } from 'next-auth';
import BookCard from '@/app/components/BookCard';
import RecentBookItem from '@/app/components/RecentBookItem';
import TabBar from '@/app/components/TabBar';
import { mockBooks, type Book } from '@/app/lib/mock-data';

interface HomePageProps {
  session: Session | null;
  books?: Book[];
  recent?: Array<{ id: number; book: Book; lastIdx: number; updatedAt: Date }>;
}

export default function HomePage({ session, books = [], recent = [] }: HomePageProps) {
  const router = useRouter();
  const [booksState, setBooksState] = useState<Book[]>(books);
  const [recentLearning, setRecentLearning] = useState<any[]>(recent);

  // Client-side refresh to avoid staleness
  useEffect(() => {
    let aborted = false;
    async function fetchAll() {
      try {
        const resBooks = await fetch('/api/books', { cache: 'no-store' });
        const jsBooks = await resBooks.json();
        if (!aborted && jsBooks?.ok) setBooksState(jsBooks.data);
      } catch {}

      if (session?.user) {
        try {
          const resRecent = await fetch('/api/recent-learning', { cache: 'no-store' });
          const jsRecent = await resRecent.json();
          if (!aborted && jsRecent?.ok) setRecentLearning(jsRecent.data);
        } catch {}
      }
    }
    fetchAll();
    return () => { aborted = true; };
  }, [session]);

  const handleBookClick = (bookId: string) => {
    if (!session) {
      // Navigate to profile page with login modal
      router.push('/me?auth=login');
    } else {
      // Navigate to learning page
      router.push(`/learn/${bookId}`);
    }
  };

  const handleContinueLearning = (bookId: string) => {
    router.push(`/learn/${bookId}`);
  };

  // 单词书卡片不再显示进度

  return (
    <div className="min-h-screen bg-gray-50 pb-20">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-10">
        <div className="px-4 py-4">
          <h1 className="text-xl font-bold text-gray-900">单词学习</h1>
        </div>
      </header>

      {/* Content */}
      <div className="px-4 py-4">
        {/* Recent Learning Section - Only show when logged in and has data */}
        {session && recentLearning.length > 0 && (
          <section className="mb-6">
            <h2 className="text-lg font-semibold text-gray-900 mb-3">最近学习</h2>
            <div className="flex gap-3 overflow-x-auto pb-2 scrollbar-hide">
              {recentLearning.map((item) => (
                <RecentBookItem
                  key={item.id}
                  book={item.book}
                  progress={{
                    id: item.id,
                    userId: 0,
                    bookId: item.book.bookId,
                    lastIdx: item.lastIdx,
                    lastWordId: undefined,
                    updatedAt: item.updatedAt,
                    createdAt: item.updatedAt,
                  } as any}
                  onClick={() => handleContinueLearning(item.book.bookId)}
                />
              ))}
            </div>
          </section>
        )}

        {/* Books Section */}
        <section>
          <h2 className="text-lg font-semibold text-gray-900 mb-3">单词书</h2>
          {(booksState.length ? booksState : mockBooks).length === 0 ? (
            <div className="text-center py-12">
              <p className="text-gray-500">暂无词书</p>
            </div>
          ) : (
            <div className="grid grid-cols-2 gap-3">
              {(booksState.length ? booksState : mockBooks).map(book => (
                <BookCard
                  key={book.id}
                  book={book}
                  onClick={() => handleBookClick(book.bookId)}
                />
              ))}
            </div>
          )}
        </section>
      </div>

      {/* TabBar */}
      <TabBar active="home" />
    </div>
  );
}
