# 项目说明（AGENTS）

本项目是一个基于 Next.js 14 + NextAuth + PostgreSQL 的 H5 英语单词学习应用，支持词书浏览、逐词学习、学习进度记录与断点续学、单词详情页以及最近学习列表。前端采用 App Router（RSC + Server Actions），后端以数据库访问 + API Route 提供数据；页面进入后再通过前端请求更新，避免数据不一致。

## 目录结构
- `app/`
  - 页面与路由：
    - 首页：`app/page.tsx`
    - 我的：`app/me/page.tsx`
    - 学习页：`app/learn/[bookId]/page.tsx`
    - 单词详情：`app/word/[bookId]/[idx]/page.tsx`
  - 组件：`app/components/*`
    - 首页：`HomePage.tsx`
    - 我的：`ProfilePage.tsx`
    - 学习：`LearningPage.tsx`、`WordCard.tsx`
    - 详情：`WordDetailPage.tsx`
    - 其他：`BookCard.tsx`、`RecentBookItem.tsx`、`TabBar.tsx`、`AuthModal.tsx`
  - API 接口：`app/api/*`
    - 书库：`app/api/books/route.ts`
    - 最近学习：`app/api/recent-learning/route.ts`
    - 我的进度：`app/api/me/progress/route.ts`
  - 服务端动作：`app/actions/progress.ts`（写入学习进度）
  - 鉴权：`app/auth.ts`、`app/auth.config.ts`、`middleware.ts`
  - 数据访问：`app/db.ts`（Drizzle schema + 原生查询）
- `db/migrations/`：数据库迁移脚本（如 `001_create_learning_schema.sql`）
- `scripts/`：迁移执行器（`run-migrations.mjs`）
- `docs/`：需求与技术文档（`docs/prd.md`、`docs/tech.md`）
- 根配置：`next.config.js`、`tailwind.config.ts`、`.eslintrc.json`、`tsconfig.json`

## 页面结构与行为
- 首页 `/`（`app/page.tsx` → `app/components/HomePage.tsx`）
  - 登录后显示“最近学习”（从 `user_book_progress` 按 `updated_at desc` 取 Top 3）
  - 展示全部词书卡片（封面、标题、词数、标签）
  - 点击词书进入 `/learn/[bookId]`，开始或继续学习
  - 首屏 SSR（服务端拉取数据），进入后通过 `/api/books` 与 `/api/recent-learning` 再刷新一次
- 我的 `/me`（`app/me/page.tsx` → `app/components/ProfilePage.tsx`）
  - 显示账户信息、学习进度列表（每本书的已学数量与百分比）与学习统计（已学单词总数、学习词书数）
  - 未登录时弹出登录/注册弹窗
  - 首屏 SSR，进入后通过 `/api/me/progress` 再刷新一次
- 学习页 `/learn/[bookId]`（`app/learn/[bookId]/page.tsx` → `LearningPage.tsx`）
  - 服务端根据 `user_book_progress.last_idx + 1` 计算起始单词 idx，并拉取该词书全部 `words`
  - 客户端渲染 `WordCard`，点击“下一个”推进；使用 Server Action 写回进度
- 单词详情 `/word/[bookId]/[idx]`（`app/word/[bookId]/[idx]/page.tsx` → `WordDetailPage.tsx`）
  - 展示 `words.content.word.content` 中的释义（多条）、例句（多条）、同近、同根、记忆法等
  - 支持有道发音播放：`https://dict.youdao.com/dictvoice?audio={word}&type={1|2}`（1=英、2=美）

## 后端接口
- GET `/api/books`
  - 描述：返回全部词书
  - 响应：`[{ id, bookId, title, wordCount, coverUrl, tags, createdAt }]`
- GET `/api/recent-learning`（需登录）
  - 描述：当前用户最近学习（Top 3）
  - 响应：`[{ id, lastIdx, updatedAt, book: {...} }]`
- GET `/api/me/progress`（需登录）
  - 描述：当前用户所有词书学习进度（用于“我的”页）
  - 响应：`[{ id, lastIdx, updatedAt, book: {...} }]`
- Server Action：`advanceProgress(bookId, currentIdx, currentWordId?)`
  - 位置：`app/actions/progress.ts`
  - 作用：更新 `user_book_progress.last_idx/last_word_id/updated_at`

## 数据库 Schema（关键表）
- 词书表 `public.books`
  - 字段：`id serial PK`、`title text`、`word_count int`、`cover_url text`、`book_id text unique`、`tags text`、`created_at timestamp default now()`
- 单词表 `public.words`
  - 字段：`id bigint PK`、`"wordRank" int`、`"headWord" text`、`content json`、`"bookId" text`
  - `content` 的结构（简化）：`{ word: { wordId, content: { trans[], sentence.sentences[], syno.synos[], relWord.rels[], ukphone, usphone, remMethod } } }`
- 学习进度表 `public.user_book_progress`
  - 字段：`id bigserial PK`、`user_id int`、`book_id text`、`last_idx int default -1`、`last_word_id bigint null`、`updated_at/created_at timestamp`
  - 约束：`unique(user_id, book_id)`；外键关联 `books(book_id)`、`words(id)`、`"User"(id)`
- 用户单词状态 `public.user_word_state`（可选，预留）
  - 字段：收藏 `starred`、难度 `difficulty`、接触次数 `exposures`、SRS 相关字段等
- 学习事件 `public.study_event`（可选）
  - 字段：`action`（view/next/open_detail/...）、`meta jsonb`、时间戳等
- 用户表 `public."User"`
  - 字段：`id serial PK`、`email varchar(64) unique`、`password varchar(64)`（示例用法，生产建议使用 NextAuth Adapter）

注：本仓库提供迁移脚本创建上述进度/状态/事件相关表（`db/migrations/001_create_learning_schema.sql`），`books`/`words` 请确保已存在或由外部导入。

## 数据访问（服务端）
- 位置：`app/db.ts`
- 能力：
  - 词书：`listBooks()`
  - 单词：`listWordsByBook(bookId)`（按 `"wordRank"` 升序）
  - 进度：`getOrInitUserBookProgress(userId, bookId)`、`getNextIndexForUser(userId, bookId)`、`updateUserBookProgress(userId, bookId, lastIdx, lastWordId)`
  - 最近学习：`listRecentLearningByUser(userId, limit)`、所有进度：`listAllLearningByUser(userId)`
  - 用户：`getUser(email)`（用于从 session.email 反查 `user_id`）

## 启动与构建
1) 准备环境变量（复制 `.env.example` 为 `.env`，并设置以下变量）：
   - `POSTGRES_URL`：PostgreSQL 连接串（生产/本地均可）
   - `AUTH_SECRET`：NextAuth 本地开发所需密钥（可用 `openssl rand -base64 32` 生成）
2) 安装依赖并运行迁移：
   - `pnpm install`
   - `pnpm db:migrate`（创建 `User` 唯一约束、进度/状态/事件表等）
3) 启动开发：`pnpm dev` → 打开 `http://localhost:3000`
4) 生产构建与运行：
   - 构建：`pnpm build`
   - 启动：`pnpm start`
5) 代码检查：`pnpm lint`

## 脚本命令（package.json）
- `dev`：启动开发服务器（Turbo 模式）
- `build`：构建生产产物
- `start`：本地运行生产构建
- `lint`：ESLint 检查
- `db:migrate`：执行 `db/migrations` 下的 SQL 迁移（脚本见 `scripts/run-migrations.mjs`）

## 其他说明
- UI 风格：现代简约、蓝色主题，移动端优先，底部固定 TabBar（首页/我的）。
- 数据一致性：页面首屏 SSR 以提升首屏速度，进入后会通过前端调用 API 刷新数据，确保与后端一致。
- 发音：使用有道接口播放英/美音标录音。
