import React from 'react';
import { withErrorBoundary, withSuspense } from '@extension/shared';
import '@src/Popup.css';

// Inline styles for the popup
const styles = {
  container: {
    width: '300px',
    padding: '16px',
    display: 'flex',
    flexDirection: 'column' as const,
    alignItems: 'center',
    backgroundColor: '#f8f9fa',
    borderRadius: '8px',
    fontFamily: 'Arial, sans-serif',
  },
  header: {
    width: '100%',
    display: 'flex',
    flexDirection: 'column' as const,
    alignItems: 'center',
    marginBottom: '16px',
  },
  title: {
    fontSize: '18px',
    fontWeight: 'bold',
    marginBottom: '8px',
    color: '#333',
    textAlign: 'center' as const,
  },
  subtitle: {
    fontSize: '14px',
    color: '#666',
    marginBottom: '16px',
    textAlign: 'center' as const,
  },
  button: {
    padding: '10px 16px',
    backgroundColor: '#4285f4',
    color: 'white',
    border: 'none',
    borderRadius: '4px',
    fontSize: '14px',
    fontWeight: 'bold',
    cursor: 'pointer',
    transition: 'all 0.3s ease',
    boxShadow: '0 2px 4px rgba(0, 0, 0, 0.1)',
  },
  buttonDisabled: {
    backgroundColor: '#ccc',
    cursor: 'not-allowed',
    boxShadow: 'none',
  },
  buttonHover: {
    backgroundColor: '#3367d6',
    boxShadow: '0 4px 8px rgba(0, 0, 0, 0.2)',
  },
  status: {
    marginTop: '12px',
    fontSize: '12px',
    color: '#666',
    fontStyle: 'italic',
  },
};

const notificationOptions = {
  type: 'basic',
  iconUrl: chrome.runtime.getURL('icon-34.png'),
  title: 'Scraping error',
  message: 'You can only use this on the Coze template page!',
} as const;

const Popup = () => {
  const [isHovering, setIsHovering] = React.useState(false);
  const [status, setStatus] = React.useState('');
  
  const scrapeCozeData = async () => {
    const [tab] = await chrome.tabs.query({ currentWindow: true, active: true });
    
    // Check if we're on the Coze template page
    if (!tab.url || !tab.url.includes('coze.cn/template')) {
      chrome.notifications.create('scrape-error', notificationOptions);
      setStatus('错误：请在 Coze 模板页面上使用此功能');
      return;
    }
    
    setStatus('正在获取数据...');
    
    // Execute the scraping script
    await chrome.scripting
      .executeScript({
        target: { tabId: tab.id! },
        func: scrapeAndDownload,
      })
      .then(result => {
        if (result && result[0] && result[0].result && result[0].result.success) {
          setStatus(`成功获取数据！已下载 ${result[0].result.count} 个模板`);
        } else {
          setStatus('未找到可抓取的数据');
        }
      })
      .catch(err => {
        console.error('Error executing script:', err);
        chrome.notifications.create('scrape-error', {
          ...notificationOptions,
          message: 'Error scraping data: ' + err.message
        });
        setStatus('获取数据时出错');
      });
  };

  // Check if we're on the right page to enable/disable the button
  const [isCozeTemplatePage, setIsCozeTemplatePage] = React.useState(false);
  
  React.useEffect(() => {
    const checkUrl = async () => {
      const [tab] = await chrome.tabs.query({ currentWindow: true, active: true });
      setIsCozeTemplatePage(tab.url?.includes('coze.cn/template') || false);
      
      if (tab.url?.includes('coze.cn/template')) {
        setStatus('准备就绪，点击按钮获取数据');
      } else {
        setStatus('请导航到 Coze 模板页面使用此功能');
      }
    };
    
    checkUrl();
  }, []);

  return (
    <div style={styles.container}>
      <header style={styles.header}>
        <h2 style={styles.title}>Coze 模板数据抓取工具</h2>
        <p style={styles.subtitle}>一键获取 Coze 模板数据并下载为 CSV 文件</p>
        <button
          style={{
            ...styles.button,
            ...(isHovering && !isCozeTemplatePage ? {} : isHovering ? styles.buttonHover : {}),
            ...(!isCozeTemplatePage ? styles.buttonDisabled : {})
          }}
          disabled={!isCozeTemplatePage}
          onClick={scrapeCozeData}
          onMouseEnter={() => setIsHovering(true)}
          onMouseLeave={() => setIsHovering(false)}>
          获取 Coze 数据
        </button>
        {status && <p style={styles.status}>{status}</p>}
      </header>
    </div>
  );
};

// This function will run in the context of the page
function scrapeAndDownload() {
  // Define the template data interface
  interface TemplateData {
    bgImg: string | null;
    appType: string | null;
    title: string | null;
    author: string | null;
    desc: string | null;
    price: string | null;
    copyCount: string | null;
  }

  function scrapeTemplateCards(): TemplateData[] {
    // Find all template cards by their structure (article elements within a flex container)
    const cards = document.querySelectorAll('div.flex.grow article');
    
    const results: TemplateData[] = [];
    
    cards.forEach(card => {
      // Extract data using positional selectors
      const result: TemplateData = {
        bgImg: null,
        appType: null,
        title: null,
        author: null,
        desc: null,
        price: null,
        copyCount: null,
      };
      
      // 1. Background image (first img within the card's image container)
      const imgElement = card.querySelector('div[class*="relative"] img');
      result.bgImg = imgElement ? imgElement.getAttribute('src') : null;
      
      // 2. App type (text in the tag after the title)
      const appTypeElement = card.querySelector('div[class*="semi-tag"] span');
      result.appType = appTypeElement && appTypeElement.textContent ? appTypeElement.textContent.trim() : null;
      
      // 3. Title (first span in the item header)
      const titleElement = card.querySelector('div > div.flex.items-center > span > span');
      result.title = titleElement && titleElement.textContent ? titleElement.textContent.trim() : null;
      
      // 4. Author (username element)
      const authorElement = card.querySelector('span[class*="name"] > span');
      result.author = authorElement && authorElement.textContent ? authorElement.textContent.trim() : null;
      
      // 5. Description (span with multiple lines in the content area)
      const descElement = card.querySelector('span[class*="semi-typography-ellipsis-multiple-line"] > span');
      result.desc = descElement && descElement.textContent ? descElement.textContent.trim() : null;
      
      // 6. Price (div in the footer area that contains the price)
      const priceElement = card.querySelector('div.mt-\\[4px\\] > div.font-medium');
      if (priceElement && priceElement.textContent) {
        const priceText = priceElement.textContent.trim();
        // 如果是免费，显示为0；如果有¥符号，移除
        if (priceText === '免费') {
          result.price = '0';
        } else {
          // 移除价格中的¥符号
          result.price = priceText.replace('¥', '');
        }
      } else {
        result.price = null;
      }
      
      // 7. Copy count (number in the footer near "复制")
      const copyCountContainer = card.querySelector('div.flex.items-center.gap-\\[4px\\]');
      if (copyCountContainer) {
        const countSpan = copyCountContainer.querySelector('span');
        if (countSpan && countSpan.textContent) {
          const countText = countSpan.textContent.trim();
          // 处理带有K的数字，如"1.2K" -> "1200"
          if (countText.endsWith('K') || countText.endsWith('k')) {
            const numValue = parseFloat(countText.replace(/[Kk]$/, ''));
            result.copyCount = String(Math.round(numValue * 1000));
          } else {
            result.copyCount = countText;
          }
        } else {
          result.copyCount = null;
        }
      }
      
      results.push(result);
    });
    
    return results;
  }

  // Convert JSON to CSV
  function convertToCSV(objArray: TemplateData[]): string {
    const array = typeof objArray !== 'object' ? JSON.parse(objArray as unknown as string) : objArray;
    let csv = '';

    // Add headers
    const headers = Object.keys(array[0]);
    csv += headers.join(',') + '\r\n';

    // Add rows
    for (let i = 0; i < array.length; i++) {
      let line = '';
      for (const index in array[i]) {
        if (line !== '') line += ',';
        
        // Handle CSV special characters
        let value = array[i][index as keyof TemplateData];
        if (value === null || value === undefined) {
          value = '';
        } else if (typeof value === 'string') {
          // Escape quotes and wrap in quotes if contains commas or quotes
          if (value.includes(',') || value.includes('"') || value.includes('\n')) {
            value = '"' + value.replace(/"/g, '""') + '"';
          }
        }
        line += value;
      }
      csv += line + '\r\n';
    }
    return csv;
  }

  // Download data as CSV
  function downloadCSV(csvContent: string): void {
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.setAttribute('href', url);
    link.setAttribute('download', 'coze_template_data.csv');
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  }

  // Execute the scraping
  const templateData = scrapeTemplateCards();
  
  // Check if we have data
  if (templateData && templateData.length > 0) {
    const csvContent = convertToCSV(templateData);
    downloadCSV(csvContent);
    return { success: true, count: templateData.length };
  } else {
    return { success: false, message: 'No template data found' };
  }
}

export default withErrorBoundary(withSuspense(Popup, <div>Loading...</div>), <div>Error Occurred</div>);
