export interface AITool {
  id: string
  name: string
  slug: string
  description: string
  full_description?: string
  website_url?: string
  image_url?: string
  logo_url?: string
  category_id: string
  created_at: string
  updated_at: string
  user_id?: string
  is_approved: boolean
  type?: string
}

export interface ToolSubmission {
  id: string
  name: string
  slug: string
  description: string
  full_description?: string
  website_url: string
  image_url?: string
  logo_url?: string
  category_id: string
  user_id: string
  status: 'pending' | 'approved' | 'rejected'
  admin_comments?: string
  created_at: string
  updated_at: string
  processed_at?: string
  ai_tool_id?: string
  type?: string
}

export async function getAllTools(): Promise<AITool[]> {
  try {
    const response = await fetch('/api/tools', {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.message || 'Failed to fetch tools');
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error('Failed to fetch tools:', error);
    return [];
  }
}

export async function getToolsByCategory(categoryId: string): Promise<AITool[]> {
  try {
    const response = await fetch(`/api/tools/category/${categoryId}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.message || 'Failed to fetch tools by category');
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error(`Failed to fetch tools for category ${categoryId}:`, error);
    return [];
  }
}

export async function getToolBySlug(slug: string): Promise<AITool | null> {
  try {
    const response = await fetch(`/api/tools/${slug}`, {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json',
      },
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.message || 'Failed to fetch tool details');
    }

    const data = await response.json();
    return data;
  } catch (error) {
    console.error(`Failed to fetch tool with slug ${slug}:`, error);
    return null;
  }
}

export async function submitTool(toolData: Omit<ToolSubmission, 'id' | 'created_at' | 'updated_at' | 'status' | 'user_id'>): Promise<{ success: boolean; error?: string; data?: ToolSubmission }> {
  try {
    const response = await fetch('/api/tools/submit', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(toolData),
    });

    const data = await response.json();

    if (!response.ok) {
      throw new Error(data.error || 'Failed to submit tool');
    }

    return { success: true, data: data };
  } catch (error: any) {
    console.error('Failed to submit tool:', error);
    return { success: false, error: error.message || 'An unknown error occurred' };
  }
} 