import { createClient } from './client'
import { ToolSubmission } from './tools'

/**
 * 获取当前用户的工具提交记录
 * @returns 用户的提交记录列表
 */
export async function getUserSubmissions(): Promise<ToolSubmission[]> {
  const supabase = createClient()
  
  // 验证用户是否已登录
  const { data: { session } } = await supabase.auth.getSession()
  if (!session) {
    throw new Error('User not authenticated')
  }
  
  // 获取当前用户的提交记录
  const { data, error } = await supabase
    .from('tool_submissions')
    .select('*, categories(name)')
    .eq('user_id', session.user.id)
    .order('created_at', { ascending: false })
  
  if (error) {
    console.error('Error fetching user submissions:', error)
    throw error
  }
  
  return data || []
}

/**
 * 获取单个提交记录详情
 * @param id 提交记录ID
 * @returns 提交记录详情
 */
export async function getSubmissionById(id: string): Promise<ToolSubmission | null> {
  const supabase = createClient()
  
  // 验证用户是否已登录
  const { data: { session } } = await supabase.auth.getSession()
  if (!session) {
    throw new Error('User not authenticated')
  }
  
  // 获取指定的提交记录，并确保它属于当前用户
  const { data, error } = await supabase
    .from('tool_submissions')
    .select('*, categories(name)')
    .eq('id', id)
    .eq('user_id', session.user.id)
    .single()
  
  if (error) {
    if (error.code === 'PGRST116') {
      return null
    }
    console.error('Error fetching submission details:', error)
    throw error
  }
  
  return data
} 