"use client"

import type React from "react"

import { useState } from "react"
import Image from "next/image"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Upload, X } from "lucide-react"
import { useToast } from "@/components/ui/use-toast"

interface FileUploaderProps {
  value: string
  onChange: (value: string) => void
  onFileSelect?: (file: File) => Promise<void>
  accept?: string
  maxSize?: number
  previewWidth?: number
  previewHeight?: number
  label?: string
}

export function FileUploader({
  value,
  onChange,
  onFileSelect,
  accept = "image/*",
  maxSize = 2 * 1024 * 1024, // 2MB
  previewWidth = 200,
  previewHeight = 200,
  label = "选择文件",
}: FileUploaderProps) {
  const { toast } = useToast()
  const [isDragging, setIsDragging] = useState(false)
  const [isUploading, setIsUploading] = useState(false)

  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0]
    if (!file) return

    if (file.size > maxSize) {
      toast({
        title: "文件过大",
        description: `文件大小不能超过 ${maxSize / (1024 * 1024)}MB`,
        variant: "destructive",
      })
      return
    }

    if (onFileSelect) {
      try {
        setIsUploading(true)
        await onFileSelect(file)
      } catch (error) {
        console.error('Upload error:', error)
        toast({
          title: "上传失败",
          description: "文件上传过程中发生错误",
          variant: "destructive",
        })
      } finally {
        setIsUploading(false)
      }
    } else {
      // 本地预览模式
      const reader = new FileReader()
      reader.onload = () => {
        onChange(reader.result as string)
      }
      reader.readAsDataURL(file)
    }
  }

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault()
    setIsDragging(true)
  }

  const handleDragLeave = () => {
    setIsDragging(false)
  }

  const handleDrop = async (e: React.DragEvent) => {
    e.preventDefault()
    setIsDragging(false)

    const file = e.dataTransfer.files[0]
    if (!file) return

    if (file.size > maxSize) {
      toast({
        title: "文件过大",
        description: `文件大小不能超过 ${maxSize / (1024 * 1024)}MB`,
        variant: "destructive",
      })
      return
    }

    if (onFileSelect) {
      try {
        setIsUploading(true)
        await onFileSelect(file)
      } catch (error) {
        console.error('Upload error:', error)
        toast({
          title: "上传失败",
          description: "文件上传过程中发生错误",
          variant: "destructive",
        })
      } finally {
        setIsUploading(false)
      }
    } else {
      // 本地预览模式
      const reader = new FileReader()
      reader.onload = () => {
        onChange(reader.result as string)
      }
      reader.readAsDataURL(file)
    }
  }

  const handleRemove = () => {
    onChange("")
  }

  const inputId = `file-upload-${Math.random().toString(36).substr(2, 9)}`

  return (
    <div>
      {value ? (
        <div className="relative">
          <div
            style={{
              width: previewWidth,
              height: previewHeight,
              position: "relative",
            }}
            className="border rounded-md overflow-hidden"
          >
            <Image src={value || "/placeholder.svg"} alt="Preview" fill className="object-cover" />
          </div>
          <Button
            type="button"
            variant="destructive"
            size="icon"
            className="absolute top-2 right-2 h-6 w-6"
            onClick={handleRemove}
          >
            <X className="h-4 w-4" />
          </Button>
        </div>
      ) : (
        <div
          className={`border-2 border-dashed rounded-md p-6 text-center ${
            isDragging ? "border-primary bg-primary/10" : "border-muted-foreground/20"
          }`}
          onDragOver={handleDragOver}
          onDragLeave={handleDragLeave}
          onDrop={handleDrop}
        >
          <Upload className="h-10 w-10 text-muted-foreground mx-auto mb-4" />
          <p className="text-sm text-muted-foreground mb-2">拖放文件到此处，或</p>
          <Input type="file" accept={accept} onChange={handleFileChange} className="hidden" id={inputId} disabled={isUploading} />
          <Button
            type="button"
            variant="outline"
            size="sm"
            onClick={() => document.getElementById(inputId)?.click()}
            disabled={isUploading}
          >
            {isUploading ? "上传中..." : label}
          </Button>
        </div>
      )}
    </div>
  )
}
