"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { ScrollArea } from "@/components/ui/scroll-area"
import { Plus } from "lucide-react"
import { cn } from "@/lib/utils"
import Image from "next/image"
import { useRouter } from "next/navigation"
import { useToast } from "@/components/ui/use-toast"
import { createClient } from "@/lib/supabase/client"

interface Category {
  id: string
  name: string
  icon: string
}

interface CategorySidebarProps {
  categories: Category[]
  activeCategory?: string | null
  onCategoryClick?: (categoryId: string) => void
  onSubmitClick?: () => void
}

export default function CategorySidebar({
  categories,
  activeCategory: propActiveCategory,
  onCategoryClick,
  onSubmitClick,
}: CategorySidebarProps) {
  const router = useRouter()
  const { toast } = useToast()
  const [activeCategory, setActiveCategory] = useState<string | null>(propActiveCategory || null)
  const [isLoggedIn, setIsLoggedIn] = useState(false)

  // Check login status when component mounts
  useEffect(() => {
    async function checkAuthStatus() {
      try {
        const supabase = createClient()
        const { data } = await supabase.auth.getSession()
        setIsLoggedIn(!!data.session)
      } catch (error) {
        console.error("Error checking auth status:", error)
      }
    }
    
    checkAuthStatus()
  }, [])

  const handleCategoryClick = (categoryId: string) => {
    setActiveCategory(categoryId)
    
    if (onCategoryClick) {
      onCategoryClick(categoryId)
    } else {
      // Default behavior: scroll to category section
      document.getElementById(categoryId)?.scrollIntoView({
        behavior: "smooth",
        block: "start",
      })
    }
  }

  const handleSubmitClick = () => {
    if (onSubmitClick) {
      onSubmitClick()
    } else {
      // Default behavior: check login and redirect or show toast
      if (isLoggedIn) {
        router.push("/user/submit")
      } else {
        toast({
          title: "请先登录",
          description: "提交工具前请先登录您的账户",
        })
      }
    }
  }

  return (
    <div className="hidden md:block w-64 border-r bg-background h-[calc(100vh-4rem)] sticky top-16">
      <ScrollArea className="h-full py-4">
        <div className="px-3 py-2">
          <h2 className="mb-2 px-4 text-lg font-semibold">分类</h2>
          <div className="space-y-1">
            {categories.map((category) => (
              <Button
                key={category.id}
                variant="ghost"
                className={cn("w-full justify-start", activeCategory === category.id && "bg-accent")}
                onClick={() => handleCategoryClick(category.id)}
              >
                <div className="w-5 h-5 relative flex-shrink-0 flex align-center">
                  <Image 
                    src={category.icon} 
                    alt={category.name} 
                    width={14} 
                    height={14}
                    className="object-contain"
                  />
                </div>
                {category.name}
              </Button>
            ))}
          </div>
        </div>
        <div className="px-3 py-4 mt-4">
          <Button className="w-full flex items-center gap-2" onClick={handleSubmitClick}>
            <Plus className="h-4 w-4" />
            提交站点
          </Button>
        </div>
      </ScrollArea>
    </div>
  )
}
