"use client"

import { useState, useEffect } from "react"
import { useRouter } from "next/navigation"
import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import * as z from "zod"
import { Button } from "@/components/ui/button"
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from "@/components/ui/form"
import { Input } from "@/components/ui/input"
import { Textarea } from "@/components/ui/textarea"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"
import { useToast } from "@/components/ui/use-toast"
import { createClient } from "@/lib/supabase/client"
import { getCategories, type Category } from "@/lib/supabase/categories"
import { submitTool } from "@/lib/supabase/tools"
import { uploadToolLogo, uploadToolPreview } from "@/lib/supabase/upload"
import MarkdownContent from "@/components/markdown-content"
import { FileUploader } from "@/components/file-uploader"

// Define form validation schema
const formSchema = z.object({
  name: z.string().min(2, "工具名称至少需要2个字符").max(100, "工具名称不能超过100个字符"),
  slug: z.string().min(2, "URL标识至少需要2个字符").max(100, "URL标识不能超过100个字符")
    .regex(/^[a-z0-9-]+$/, "URL标识只能包含小写字母、数字和连字符"),
  description: z.string().min(10, "简介至少需要10个字符").max(500, "简介不能超过500个字符"),
  website_url: z.string().url("请输入有效的URL地址"),
  category_id: z.string().min(1, "请选择一个分类"),
  full_description: z.string().optional(),
  logo_url: z.string().optional(),
  image_url: z.string().optional(),
})

type FormData = z.infer<typeof formSchema>

export default function SubmitToolPage() {
  const router = useRouter()
  const { toast } = useToast()
  const [categories, setCategories] = useState<Category[]>([])
  const [isLoading, setIsLoading] = useState(false)
  const [isAuthenticated, setIsAuthenticated] = useState(false)
  const [activeTab, setActiveTab] = useState<"edit" | "preview">("edit")

  // Initialize form
  const form = useForm<FormData>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      name: "",
      slug: "",
      description: "",
      website_url: "",
      category_id: "",
      full_description: "",
      logo_url: "",
      image_url: "",
    },
  })

  // Get the current value of the full_description field for the preview
  const fullDescription = form.watch("full_description") || ""
  
  // 监听工具名称变化，自动生成slug
  const name = form.watch("name")
  useEffect(() => {
    if (name && !form.getValues("slug")) {
      const slug = name
        .toLowerCase()
        .replace(/\s+/g, "-") // 空格转连字符
        .replace(/[^\w\-]+/g, "") // 移除非字母数字和连字符
        .replace(/\-\-+/g, "-") // 替换多个连字符为单个
        .replace(/^-+/, "") // 去除开头的连字符
        .replace(/-+$/, "") // 去除结尾的连字符
      
      form.setValue("slug", slug)
    }
  }, [name, form])

  // Check if user is authenticated
  useEffect(() => {
    async function checkAuth() {
      const supabase = createClient()
      const { data } = await supabase.auth.getSession()
      
      if (!data.session) {
        // User is not authenticated, redirect to login
        toast({
          title: "需要登录",
          description: "提交工具前请先登录您的账户",
          variant: "destructive",
        })
        router.push("/login")
      } else {
        setIsAuthenticated(true)
      }
    }

    // Load categories
    async function loadCategories() {
      try {
        const data = await getCategories()
        setCategories(data)
      } catch (error) {
        console.error("Error loading categories:", error)
        toast({
          title: "加载分类失败",
          description: "请稍后再试",
          variant: "destructive",
        })
      }
    }

    checkAuth()
    loadCategories()
  }, [router, toast])

  // 处理logo文件上传
  const handleLogoUpload = async (file: File) => {
    try {
      const logoUrl = await uploadToolLogo(file)
      form.setValue("logo_url", logoUrl)
    } catch (error) {
      console.error("Error uploading logo:", error)
      toast({
        title: "上传失败",
        description: "Logo上传失败，请重试",
        variant: "destructive",
      })
    }
  }

  // 处理预览图文件上传
  const handleImageUpload = async (file: File) => {
    try {
      const imageUrl = await uploadToolPreview(file)
      form.setValue("image_url", imageUrl)
    } catch (error) {
      console.error("Error uploading image:", error)
      toast({
        title: "上传失败",
        description: "预览图上传失败，请重试",
        variant: "destructive",
      })
    }
  }

  // Handle form submission
  async function onSubmit(data: FormData) {
    if (!isAuthenticated) {
      toast({
        title: "需要登录",
        description: "提交工具前请先登录您的账户",
        variant: "destructive",
      })
      return
    }

    setIsLoading(true)

    try {
      const result = await submitTool({
        name: data.name,
        slug: data.slug,
        description: data.description,
        website_url: data.website_url,
        category_id: data.category_id,
        full_description: data.full_description,
        logo_url: data.logo_url,
        image_url: data.image_url,
      })

      if (result.success) {
        toast({
          title: "提交成功",
          description: "感谢您的提交，我们会尽快审核",
        })
        router.push("/user/profile")
      } else {
        toast({
          title: "提交失败",
          description: result.error || "请稍后重试",
          variant: "destructive",
        })
      }
    } catch (error) {
      console.error("Error submitting tool:", error)
      toast({
        title: "提交失败",
        description: "发生错误，请稍后重试",
        variant: "destructive",
      })
    } finally {
      setIsLoading(false)
    }
  }

  if (!isAuthenticated) {
    return null // Page will redirect in useEffect
  }

  return (
    <div className="container max-w-2xl mx-auto py-10 px-4">
      <h1 className="text-2xl font-bold mb-6">提交新工具</h1>
      
      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
          <FormField
            control={form.control}
            name="name"
            render={({ field }) => (
              <FormItem>
                <FormLabel>工具名称</FormLabel>
                <FormControl>
                  <Input placeholder="例如：ChatGPT" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="slug"
            render={({ field }) => (
              <FormItem>
                <FormLabel>URL标识</FormLabel>
                <FormDescription>
                  用于生成该工具的URL路径，仅允许小写字母、数字和连字符
                </FormDescription>
                <FormControl>
                  <Input placeholder="例如：chatgpt" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="description"
            render={({ field }) => (
              <FormItem>
                <FormLabel>简短描述</FormLabel>
                <FormControl>
                  <Textarea 
                    placeholder="简短描述工具的主要功能和特点（最多500字符）" 
                    {...field} 
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="website_url"
            render={({ field }) => (
              <FormItem>
                <FormLabel>网站地址</FormLabel>
                <FormControl>
                  <Input placeholder="https://example.com" {...field} />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="category_id"
            render={({ field }) => (
              <FormItem>
                <FormLabel>分类</FormLabel>
                <Select onValueChange={field.onChange} defaultValue={field.value}>
                  <FormControl>
                    <SelectTrigger>
                      <SelectValue placeholder="选择分类" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    {categories.map((category) => (
                      <SelectItem key={category.id} value={category.id}>
                        {category.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="logo_url"
            render={({ field }) => (
              <FormItem>
                <FormLabel>工具Logo</FormLabel>
                <FormDescription>
                  上传工具的Logo图片，最大2MB，支持JPG、PNG、SVG和WebP格式
                </FormDescription>
                <FormControl>
                  <FileUploader 
                    value={field.value || ''}
                    onChange={field.onChange}
                    onFileSelect={handleLogoUpload}
                    maxSize={2 * 1024 * 1024}
                    accept="image/jpeg,image/png,image/svg+xml,image/webp"
                    previewWidth={100}
                    previewHeight={100}
                    label="上传Logo"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="full_description"
            render={({ field }) => (
              <FormItem>
                <FormLabel>详细介绍（可选）</FormLabel>
                <FormDescription>
                  支持Markdown格式，可以使用标题、列表、链接等丰富的格式
                </FormDescription>
                <Tabs defaultValue="edit" onValueChange={(value) => setActiveTab(value as "edit" | "preview")}>
                  <TabsList className="mb-2">
                    <TabsTrigger value="edit">编辑</TabsTrigger>
                    <TabsTrigger value="preview">预览</TabsTrigger>
                  </TabsList>
                  <TabsContent value="edit">
                    <FormControl>
                      <Textarea 
                        placeholder="详细介绍工具的功能、使用场景和优势，支持Markdown格式" 
                        className="min-h-[200px] font-mono text-sm"
                        {...field}
                      />
                    </FormControl>
                  </TabsContent>
                  <TabsContent value="preview">
                    <div className="border rounded-md p-4 min-h-[200px] bg-muted/20">
                      {fullDescription ? (
                        <MarkdownContent content={fullDescription} />
                      ) : (
                        <p className="text-muted-foreground">预览区域为空，请先添加内容</p>
                      )}
                    </div>
                  </TabsContent>
                </Tabs>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <FormField
            control={form.control}
            name="image_url"
            render={({ field }) => (
              <FormItem>
                <FormLabel>工具预览图</FormLabel>
                <FormDescription>
                  上传工具的预览图或截图，最大5MB，支持JPG、PNG和WebP格式
                </FormDescription>
                <FormControl>
                  <FileUploader 
                    value={field.value || ''}
                    onChange={field.onChange}
                    onFileSelect={handleImageUpload}
                    maxSize={5 * 1024 * 1024}
                    accept="image/jpeg,image/png,image/webp"
                    previewWidth={300}
                    previewHeight={200}
                    label="上传预览图"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
          
          <Button type="submit" className="w-full" disabled={isLoading}>
            {isLoading ? "提交中..." : "提交工具"}
          </Button>
        </form>
      </Form>
    </div>
  )
}
