"use client"

import { useState, useEffect } from "react"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { Badge } from "@/components/ui/badge"
import { Button } from "@/components/ui/button"
import { Eye, AlertCircle } from "lucide-react"
import { useToast } from "@/components/ui/use-toast"
import { Alert, AlertDescription, AlertTitle } from "@/components/ui/alert"
import Link from "next/link"
import { useRouter } from "next/navigation"
import { getUserSubmissions } from "@/lib/supabase/submissions"
import { createClient } from "@/lib/supabase/client"
import type { ToolSubmission } from "@/lib/supabase/tools"

export default function SubmissionsPage() {
  const router = useRouter()
  const { toast } = useToast()
  const [userSubmissions, setUserSubmissions] = useState<ToolSubmission[]>([])
  const [isLoading, setIsLoading] = useState(true)
  const [error, setError] = useState<string | null>(null)

  // 获取用户提交记录
  useEffect(() => {
    async function checkAuthAndLoadSubmissions() {
      try {
        // 验证用户是否已登录
        const supabase = createClient()
        const { data: { session } } = await supabase.auth.getSession()
        
        if (!session) {
          // 用户未登录，重定向到登录页面
          toast({
            title: "需要登录",
            description: "请先登录查看您的提交记录",
            variant: "destructive",
          })
          router.push("/login")
          return
        }
        
        // 加载用户提交记录
        const submissions = await getUserSubmissions()
        setUserSubmissions(submissions)
      } catch (err) {
        console.error("Error loading submissions:", err)
        setError("加载提交记录时出错，请稍后再试")
      } finally {
        setIsLoading(false)
      }
    }
    
    checkAuthAndLoadSubmissions()
  }, [router, toast])

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleString("zh-CN", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
      hour: "2-digit",
      minute: "2-digit",
    })
  }

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "pending":
        return <Badge variant="outline">审核中</Badge>
      case "approved":
        return <Badge>已发布</Badge>
      case "rejected":
        return <Badge variant="destructive">已拒绝</Badge>
      default:
        return <Badge variant="secondary">{status}</Badge>
    }
  }

  if (isLoading) {
    return <div className="py-10 text-center">加载中...</div>
  }

  if (error) {
    return (
      <Alert variant="destructive" className="my-4">
        <AlertCircle className="h-4 w-4" />
        <AlertTitle>错误</AlertTitle>
        <AlertDescription>{error}</AlertDescription>
      </Alert>
    )
  }

  return (
    <div>
      <h2 className="text-2xl font-bold mb-6">我的提交</h2>

      {userSubmissions.length > 0 ? (
        <div className="border rounded-md">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>工具名称</TableHead>
                <TableHead>提交时间</TableHead>
                <TableHead>状态</TableHead>
                <TableHead className="text-right">操作</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {userSubmissions.map((submission) => (
                <TableRow key={submission.id}>
                  <TableCell className="font-medium">{submission.name}</TableCell>
                  <TableCell>{formatDate(submission.created_at)}</TableCell>
                  <TableCell>
                    {getStatusBadge(submission.status)}
                  </TableCell>
                  <TableCell className="text-right">
                    {submission.status === "approved" && submission.ai_tool_id ? (
                      <Button variant="outline" size="sm" asChild>
                        <Link href={`/tool/${submission.slug}`} target="_blank">
                          <Eye className="h-4 w-4 mr-1" />
                          查看
                        </Link>
                      </Button>
                    ) : (
                      "--"
                    )}
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </div>
      ) : (
        <div className="text-center py-12 border rounded-md">
          <h3 className="text-lg font-medium mb-2">暂无提交记录</h3>
          <p className="text-muted-foreground mb-6">您还没有提交过任何AI工具</p>
          <Button asChild>
            <Link href="/user/submit">提交工具</Link>
          </Button>
        </div>
      )}
    </div>
  )
}
