"use client"

import { useState, useEffect } from "react"
import { Button } from "@/components/ui/button"
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar"
import { useToast } from "@/components/ui/use-toast"
import { useRouter } from "next/navigation"
import { Separator } from "@/components/ui/separator"
import { createClient } from "@/lib/supabase/client"
import type { User } from "@supabase/supabase-js"

interface UserProfile {
  email: string;
  created_at: string;
  submissionCount: number;
}

export default function ProfilePage() {
  const router = useRouter()
  const { toast } = useToast()
  const [userProfile, setUserProfile] = useState<UserProfile | null>(null)
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    async function loadUserProfile() {
      try {
        const supabase = createClient()
        
        // Get authenticated user
        const { data: userData, error: userError } = await supabase.auth.getUser()
        
        if (userError || !userData.user) {
          // Redirect to login if not authenticated
          router.push('/auth/login')
          return
        }
        
        // Get submission count (you would need a submissions table in your database)
        // This is just a placeholder - would be replaced by a real query to your submissions table
        const submissionCount = 0
        
        setUserProfile({
          email: userData.user.email || '',
          created_at: userData.user.created_at || new Date().toISOString(),
          submissionCount
        })
      } catch (error) {
        console.error('Error loading profile:', error)
        toast({
          title: "加载用户资料失败",
          description: "请稍后再试",
          variant: "destructive"
        })
      } finally {
        setLoading(false)
      }
    }
    
    loadUserProfile()
  }, [router, toast])

  const handleLogout = async () => {
    try {
      const supabase = createClient()
      await supabase.auth.signOut()
      
      toast({
        title: "已退出登录",
        description: "您已成功退出登录",
      })
      router.push("/")
    } catch (error) {
      console.error('Error logging out:', error)
      toast({
        title: "退出登录失败",
        description: "请稍后再试",
        variant: "destructive"
      })
    }
  }

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("zh-CN", {
      year: "numeric",
      month: "long",
      day: "numeric",
    })
  }
  
  // Display loading state
  if (loading) {
    return <div className="text-center py-12">加载中...</div>
  }
  
  // Safety check
  if (!userProfile) {
    return <div className="text-center py-12">无法加载用户资料</div>
  }

  const userInitial = userProfile.email ? userProfile.email.charAt(0).toUpperCase() : '用'

  return (
    <div className="max-w-2xl">
      <div className="flex flex-col md:flex-row items-start gap-8 mb-8">
        <Avatar className="h-24 w-24">
          <AvatarFallback>{userInitial}</AvatarFallback>
        </Avatar>
        <div>
          <h2 className="text-2xl font-bold">{userProfile.email.split('@')[0]}</h2>
          <p className="text-muted-foreground">{userProfile.email}</p>
        </div>
      </div>

      <Separator className="my-6" />

      <div className="space-y-6">
        <div>
          <h3 className="text-lg font-medium mb-4">账户信息</h3>
          <dl className="space-y-4">
            <div className="flex justify-between">
              <dt className="text-muted-foreground">注册时间</dt>
              <dd>{formatDate(userProfile.created_at)}</dd>
            </div>
            <div className="flex justify-between">
              <dt className="text-muted-foreground">提交工具数</dt>
              <dd>{userProfile.submissionCount}</dd>
            </div>
          </dl>
        </div>

        <Separator className="my-6" />

        <div>
          <h3 className="text-lg font-medium mb-4">账户操作</h3>
          <Button variant="outline" onClick={handleLogout}>
            退出登录
          </Button>
        </div>
      </div>
    </div>
  )
}
