"use client"

import { useEffect, useState } from "react"
import { useSearchParams } from "next/navigation"
import { createClient } from "@/lib/supabase/client"
import ToolCard from "@/components/tool-card"
import { Input } from "@/components/ui/input"
import { Button } from "@/components/ui/button"
import { Search as SearchIcon } from "lucide-react"
import type { AITool } from "@/lib/supabase/tools"

export default function SearchPage() {
  const searchParams = useSearchParams()
  const initialQuery = searchParams.get("q") || ""
  const [searchQuery, setSearchQuery] = useState(initialQuery)
  const [searchResults, setSearchResults] = useState<AITool[]>([])
  const [isLoading, setIsLoading] = useState(false)

  useEffect(() => {
    if (initialQuery) {
      performSearch(initialQuery)
    }
  }, [initialQuery])

  const performSearch = async (query: string) => {
    if (!query.trim()) {
      setSearchResults([])
      return
    }

    setIsLoading(true)
    try {
      const supabase = createClient()
      
      // First search by name and description
      const { data: nameResults, error: nameError } = await supabase
        .from('ai_tools')
        .select('*, categories(name)')
        .eq('is_approved', true)
        .or(`name.ilike.%${query}%,description.ilike.%${query}%`)
        .order('created_at', { ascending: false })
      
      if (nameError) {
        console.error('Error searching tools:', nameError)
        return
      }
      
      // Then search in full_description if needed
      const { data: descResults, error: descError } = await supabase
        .from('ai_tools')
        .select('*, categories(name)')
        .eq('is_approved', true)
        .ilike('full_description', `%${query}%`)
        .order('created_at', { ascending: false })
      
      if (descError) {
        console.error('Error searching tools by description:', descError)
        return
      }
      
      // Combine results and remove duplicates
      const combinedResults = [...nameResults]
      
      descResults.forEach(descTool => {
        if (!combinedResults.some(tool => tool.id === descTool.id)) {
          combinedResults.push(descTool)
        }
      })
      
      setSearchResults(combinedResults)
    } catch (error) {
      console.error('Error during search:', error)
    } finally {
      setIsLoading(false)
    }
  }

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault()
    performSearch(searchQuery)
    
    // Update URL with search query for shareable links
    const url = new URL(window.location.href)
    url.searchParams.set('q', searchQuery)
    window.history.pushState({}, '', url.toString())
  }

  return (
    <div className="container mx-auto p-6">
      <h1 className="text-2xl font-bold mb-6">搜索AI工具</h1>
      
      <form onSubmit={handleSearch} className="mb-8 flex gap-2">
        <div className="relative flex-1">
          <SearchIcon className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
          <Input
            type="search"
            placeholder="输入关键词搜索工具..."
            className="pl-8"
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
          />
        </div>
        <Button type="submit" disabled={isLoading}>
          {isLoading ? "搜索中..." : "搜索"}
        </Button>
      </form>

      {searchResults.length > 0 ? (
        <div>
          <p className="mb-4">找到 {searchResults.length} 个相关工具</p>
          <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
            {searchResults.map((tool) => (
              <ToolCard key={tool.id} tool={tool} />
            ))}
          </div>
        </div>
      ) : initialQuery ? (
        <div className="text-center py-10">
          <p className="text-lg">未找到与 "{initialQuery}" 相关的工具</p>
          <p className="text-muted-foreground mt-2">请尝试其他关键词</p>
        </div>
      ) : (
        <div className="text-center py-10">
          <p className="text-lg">输入关键词开始搜索</p>
          <p className="text-muted-foreground mt-2">可以搜索工具名称、描述等</p>
        </div>
      )}
    </div>
  )
} 