import { Suspense } from "react"
import { Search } from "lucide-react"
import CategorySidebar from "@/components/category-sidebar"
import ToolCard from "@/components/tool-card"
import { createClient } from "@/lib/supabase/server"

async function getToolsAndCategories() {
  const supabase = await createClient()
  
  // Fetch categories
  const { data: categories, error: categoriesError } = await supabase
    .from('categories')
    .select('id, name, icon')
    .order('created_at', { ascending: true })

  if (categoriesError) {
    console.error('Error fetching categories:', categoriesError)
    throw new Error('Failed to fetch categories')
  }

  // Fetch all approved tools
  const { data: tools, error: toolsError } = await supabase
    .from('ai_tools')
    .select('*, categories(name)')
    .eq('is_approved', true)
    .order('created_at', { ascending: false })

  if (toolsError) {
    console.error('Error fetching tools:', toolsError)
    throw new Error('Failed to fetch tools')
  }

  // Find hot tools based on type field
  const hotTools = tools.filter(tool => tool.type === 'hot')

  // Find new tools based on type field
  const newTools = tools.filter(tool => tool.type === 'new')

  // Organize tools by category
  const toolsByCategory: Record<string, any[]> = {}
  
  for (const category of categories) {
    const categoryTools = tools.filter(tool => tool.category_id === category.id)
    if (categoryTools.length > 0) {
      toolsByCategory[category.id] = categoryTools
    }
  }

  return {
    categories,
    hotTools,
    newTools,
    toolsByCategory
  }
}

export default async function HomePage() {
  const { categories, hotTools, newTools, toolsByCategory } = await getToolsAndCategories()

  return (
    <div className="flex">
      {/* 左侧分类栏 */}
      <Suspense fallback={<div>Loading categories...</div>}>
        <CategorySidebar categories={categories} />
      </Suspense>

      {/* 主内容区 */}
      <div className="flex-1 p-6 overflow-auto">
        {/* 移动端搜索框 */}
        <div className="relative mb-6 md:hidden">
          <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
          <input
            type="search"
            placeholder="搜索AI工具..."
            className="w-full rounded-md border border-input bg-background pl-8 pr-4 py-2 text-sm ring-offset-background focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring"
          />
        </div>

        {/* 热门工具 */}
        {hotTools.length > 0 && (
          <section className="mb-10">
            <div className="mb-4">
              <h2 className="text-xl font-bold">热门工具</h2>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
              {hotTools.map((tool) => (
                <ToolCard key={tool.id} tool={tool} />
              ))}
            </div>
          </section>
        )}

        {/* 最新收录 */}
        {newTools.length > 0 && (
          <section className="mb-10">
            <div className="mb-4">
              <h2 className="text-xl font-bold">最新收录</h2>
            </div>
            <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
              {newTools.map((tool) => (
                <ToolCard key={tool.id} tool={tool} />
              ))}
            </div>
          </section>
        )}

        {/* 分类展示区 */}
        {categories.map((category) => (
          // 只显示有工具的分类
          toolsByCategory[category.id]?.length > 0 && (
            <section key={category.id} id={category.id} className="mb-10">
              <div className="mb-4">
                <h2 className="text-xl font-bold">{category.name}</h2>
              </div>
              <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {toolsByCategory[category.id]?.map((tool) => (
                  <ToolCard key={tool.id} tool={tool} />
                ))}
              </div>
            </section>
          )
        ))}
      </div>
    </div>
  )
} 