"use client"

import { useState } from "react"
import Link from "next/link"
import { Button } from "@/components/ui/button"
import { Input } from "@/components/ui/input"
import { Label } from "@/components/ui/label"
import { useToast } from "@/components/ui/use-toast"
import { createClient } from "@/lib/supabase/client"

export default function ForgotPasswordPage() {
  const { toast } = useToast()
  const [email, setEmail] = useState("")
  const [loading, setLoading] = useState(false)
  const [submitted, setSubmitted] = useState(false)

  const handleResetPassword = async (e: React.FormEvent) => {
    e.preventDefault()
    setLoading(true)

    try {
      const supabase = createClient()
      const { error } = await supabase.auth.resetPasswordForEmail(email, {
        redirectTo: `${window.location.origin}/auth/update-password`,
      })

      if (error) {
        throw error
      }

      setSubmitted(true)
    } catch (error: any) {
      toast({
        title: "重置请求失败",
        description: error.message || "发送重置密码邮件时出现错误",
        variant: "destructive",
      })
    } finally {
      setLoading(false)
    }
  }

  return (
    <div className="flex min-h-screen flex-col items-center justify-center py-12 px-4 sm:px-6 lg:px-8">
      <div className="w-full max-w-md space-y-8">
        {!submitted ? (
          <>
            <div className="text-center">
              <h1 className="text-2xl font-bold">重置密码</h1>
              <p className="mt-2 text-sm text-muted-foreground">
                输入您的邮箱地址，我们将发送重置密码链接
              </p>
            </div>

            <form className="mt-8 space-y-6" onSubmit={handleResetPassword}>
              <div>
                <Label htmlFor="email">邮箱</Label>
                <Input
                  id="email"
                  name="email"
                  type="email"
                  value={email}
                  onChange={(e) => setEmail(e.target.value)}
                  placeholder="your@email.com"
                  required
                  className="mt-1"
                />
              </div>

              <Button type="submit" className="w-full" disabled={loading}>
                {loading ? "发送中..." : "发送重置链接"}
              </Button>

              <div className="text-center text-sm">
                <Link href="/auth/login" className="text-primary hover:underline">
                  返回登录
                </Link>
              </div>
            </form>
          </>
        ) : (
          <div className="text-center">
            <h1 className="text-2xl font-bold">邮件已发送</h1>
            <p className="mt-4 text-sm text-muted-foreground">
              我们已向您的邮箱 {email} 发送了一封包含重置密码链接的邮件。
              请检查您的收件箱（以及垃圾邮件文件夹）并点击邮件中的链接重置密码。
            </p>
            <div className="mt-8">
              <Button asChild variant="outline">
                <Link href="/auth/login">返回登录</Link>
              </Button>
            </div>
          </div>
        )}
      </div>
    </div>
  )
} 