import { NextResponse } from "next/server";
import { createClient } from "@/lib/supabase/server";

export async function POST(request: Request) {
  try {
    // Parse the request body
    const body = await request.json();

    // Get required fields from the body
    const {
      name,
      description,
      website_url,
      category_id,
      full_description,
      image_url,
      slug,
      logo_url,
    } = body;

    // Validate required fields
    if (!name || !description || !website_url || !category_id) {
      return NextResponse.json(
        { error: "Missing required fields" },
        { status: 400 }
      );
    }

    // Create a Supabase client with the user's session
    const supabase = await createClient();

    // Check if the user is authenticated
    const {
      data: { user },
      error: authError,
    } = await supabase.auth.getUser();

    if (authError || !user) {
      return NextResponse.json(
        { error: "Authentication required" },
        { status: 401 }
      );
    }

    // Prepare the submission data
    const submissionData = {
      name,
      slug,
      description,
      website_url,
      category_id,
      user_id: user.id,
      status: "pending",
      full_description: full_description || null,
      image_url: image_url || null,
      logo_url: logo_url || null,
    };

    // Insert the submission to the tool_submissions table
    const { data, error } = await supabase
      .from("tool_submissions")
      .insert(submissionData)
      .select()
      .single();

    if (error) {
      console.error("Error submitting tool:", error);
      return NextResponse.json({ error: error.message }, { status: 500 });
    }

    return NextResponse.json(data, { status: 201 });
  } catch (error: any) {
    console.error("Failed to submit tool:", error);
    return NextResponse.json(
      { error: error.message || "Failed to submit tool" },
      { status: 500 }
    );
  }
}
