#!/usr/bin/env node

// 此脚本用于在Supabase中创建必要的存储桶
const { createClient } = require('@supabase/supabase-js');

// 从环境变量中获取Supabase配置
const supabaseUrl = process.env.NEXT_PUBLIC_SUPABASE_URL;
const supabaseServiceKey = process.env.SUPABASE_SERVICE_KEY;

if (!supabaseUrl || !supabaseServiceKey) {
  console.error('请确保设置了 NEXT_PUBLIC_SUPABASE_URL 和 SUPABASE_SERVICE_KEY 环境变量');
  process.exit(1);
}

// 创建admin客户端
const supabase = createClient(supabaseUrl, supabaseServiceKey, {
  auth: {
    autoRefreshToken: false,
    persistSession: false
  }
});

// 要创建的存储桶配置
const bucketsToCreate = [
  {
    name: 'tool-logos',
    public: true,
    fileSizeLimit: 2 * 1024 * 1024, // 2MB
    allowedMimeTypes: ['image/jpeg', 'image/png', 'image/svg+xml', 'image/webp']
  },
  {
    name: 'tool-previews',
    public: true,
    fileSizeLimit: 5 * 1024 * 1024, // 5MB
    allowedMimeTypes: ['image/jpeg', 'image/png', 'image/webp']
  }
];

// 创建存储桶函数
async function createBucket(bucketConfig) {
  try {
    // 检查存储桶是否已存在
    const { data: existingBuckets } = await supabase.storage.listBuckets();
    const bucketExists = existingBuckets.some(bucket => bucket.name === bucketConfig.name);

    if (bucketExists) {
      console.log(`存储桶 ${bucketConfig.name} 已存在，跳过创建`);
      return;
    }

    // 创建存储桶
    const { error } = await supabase.storage.createBucket(bucketConfig.name, {
      public: bucketConfig.public
    });

    if (error) {
      throw new Error(`创建存储桶 ${bucketConfig.name} 失败: ${error.message}`);
    }

    console.log(`成功创建存储桶: ${bucketConfig.name}`);

    // 设置存储桶策略
    if (bucketConfig.public) {
      const { error: policyError } = await supabase.storage.from(bucketConfig.name).setPublic(true);
      
      if (policyError) {
        console.warn(`设置存储桶 ${bucketConfig.name} 公开访问权限失败: ${policyError.message}`);
      }
    }

  } catch (error) {
    console.error(`处理存储桶 ${bucketConfig.name} 时出错:`, error);
  }
}

// 主函数
async function main() {
  console.log('开始创建Supabase存储桶...');

  for (const bucketConfig of bucketsToCreate) {
    await createBucket(bucketConfig);
  }

  console.log('存储桶创建完成!');
}

// 执行主函数
main().catch(error => {
  console.error('脚本执行失败:', error);
  process.exit(1);
}); 