import { createAdminClient } from "@/utils/supabase/admin";
import { createServerClient } from "@supabase/ssr";
import { type NextRequest, NextResponse } from "next/server";

export async function middleware(request: NextRequest) {
  try {
    // Create an unmodified response
    let response = NextResponse.next({
      request: {
        headers: request.headers,
      },
    });

    const supabase = createServerClient(
      process.env.NEXT_PUBLIC_SUPABASE_URL!,
      process.env.NEXT_PUBLIC_SUPABASE_ANON_KEY!,
      {
        cookies: {
          getAll() {
            return request.cookies.getAll();
          },
          setAll(cookiesToSet) {
            cookiesToSet.forEach(({ name, value }) =>
              request.cookies.set(name, value),
            );
            response = NextResponse.next({
              request,
            });
            cookiesToSet.forEach(({ name, value, options }) =>
              response.cookies.set(name, value, options),
            );
          },
        },
      },
    );

    // 刷新会话
    const { data: { user }, error: userError } = await supabase.auth.getUser();

    // 处理普通的保护路由
    if (request.nextUrl.pathname.startsWith("/protected") && userError) {
      return NextResponse.redirect(new URL("/sign-in", request.url));
    }

    // 处理管理员路由的特殊保护
    if (request.nextUrl.pathname.startsWith("/admin")) {
      // 如果未登录，重定向到登录页
      if (userError || !user) {
        const redirectUrl = new URL("/sign-in", request.url);
        redirectUrl.searchParams.set("message", "请先登录");
        return NextResponse.redirect(redirectUrl);
      }

      // 检查用户是否为管理员
      const adminClient = createAdminClient();
      const { data: adminData, error: adminError } = await adminClient
        .from("admin_users")
        .select("user_id")
        .eq("user_id", user.id)
        .single();

      // 如果不是管理员，重定向到首页
      if (adminError || !adminData) {
        const redirectUrl = new URL("/", request.url);
        redirectUrl.searchParams.set("message", "您没有管理员权限");
        return NextResponse.redirect(redirectUrl);
      }
    }

    return response;
  } catch (e) {
    console.error("中间件出错:", e);
    // 如果出错，只处理基本响应，不进行权限验证
    return NextResponse.next({
      request: {
        headers: request.headers,
      },
    });
  }
}

export const config = {
  matcher: [
    // 匹配所有请求路径，但排除静态资源
    "/((?!_next/static|_next/image|favicon.ico|.*\\.(?:svg|png|jpg|jpeg|gif|webp)$).*)",
  ],
};
