"use client";

import { useState } from "react";
import { useToast } from "@/components/ui/use-toast";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Loader2, Upload, Check, X } from "lucide-react";
import Image from "next/image";

interface FileUploaderProps {
  bucketName: string;
  accept?: string;
  maxSizeMB?: number;
  value?: string;
  onUploadComplete: (url: string) => void;
}

export function FileUploader({
  bucketName,
  accept = "image/*",
  maxSizeMB = 5,
  value,
  onUploadComplete
}: FileUploaderProps) {
  const { toast } = useToast();
  const [isUploading, setIsUploading] = useState(false);
  const [fileName, setFileName] = useState<string>("");
  const [previewUrl, setPreviewUrl] = useState<string>(value || "");
  const [useNextImage, setUseNextImage] = useState(true);
  
  const maxSizeBytes = maxSizeMB * 1024 * 1024;
  
  const handleFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;
    
    // 验证文件类型
    if (!file.type.match(accept.replace(/,/g, "|").replace(/\*/g, ".*"))) {
      toast({
        title: "错误",
        description: "不支持的文件类型",
        variant: "destructive",
      });
      return;
    }
    
    // 验证文件大小
    if (file.size > maxSizeBytes) {
      toast({
        title: "错误",
        description: `文件大小不能超过 ${maxSizeMB}MB`,
        variant: "destructive",
      });
      return;
    }
    
    setFileName(file.name);
    setIsUploading(true);
    
    try {
      // 创建表单数据上传
      const formData = new FormData();
      formData.append("file", file);
      formData.append("bucket", bucketName);
      
      const response = await fetch("/api/upload", {
        method: "POST",
        body: formData,
      });
      
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || "上传失败");
      }
      
      const { url } = await response.json();
      setPreviewUrl(url);
      onUploadComplete(url);
      
      toast({
        title: "成功",
        description: "文件上传成功",
      });
    } catch (error) {
      console.error("上传文件失败:", error);
      toast({
        title: "错误",
        description: error instanceof Error ? error.message : "上传文件失败",
        variant: "destructive",
      });
    } finally {
      setIsUploading(false);
    }
  };
  
  const clearUpload = () => {
    setPreviewUrl("");
    setFileName("");
    onUploadComplete("");
  };
  
  const handleImageError = () => {
    console.warn("Next.js Image组件加载失败，回退到标准img标签");
    setUseNextImage(false);
  };
  
  return (
    <div className="space-y-4">
      {!previewUrl ? (
        <div className="grid w-full max-w-sm items-center gap-1.5">
          <Label htmlFor="file-upload" className="sr-only">
            上传文件
          </Label>
          <Input
            id="file-upload"
            type="file"
            accept={accept}
            onChange={handleFileChange}
            disabled={isUploading}
            className="hidden"
          />
          <Button
            type="button"
            onClick={() => document.getElementById("file-upload")?.click()}
            variant="outline"
            disabled={isUploading}
            className="w-full h-24 flex flex-col items-center justify-center border-dashed"
          >
            {isUploading ? (
              <>
                <Loader2 className="h-6 w-6 animate-spin mb-2" />
                <span>上传中...</span>
              </>
            ) : (
              <>
                <Upload className="h-6 w-6 mb-2" />
                <span>点击上传文件</span>
              </>
            )}
          </Button>
        </div>
      ) : (
        <div className="relative w-full max-w-sm border rounded-md p-2">
          <div className="flex items-center justify-between mb-2">
            <span className="text-sm font-medium truncate max-w-[240px]">
              {fileName || "已上传文件"}
            </span>
            <Button
              type="button"
              variant="ghost"
              size="icon"
              onClick={clearUpload}
              className="h-8 w-8"
            >
              <X className="h-4 w-4" />
            </Button>
          </div>
          
          {previewUrl.match(/\.(jpg|jpeg|png|gif|webp)$/i) && (
            <div className="relative h-32 w-full bg-muted rounded-md overflow-hidden">
              {useNextImage ? (
                <Image
                  src={previewUrl}
                  alt="上传预览"
                  fill
                  className="object-contain"
                  onError={handleImageError}
                />
              ) : (
                <img
                  src={previewUrl}
                  alt="上传预览"
                  className="w-full h-full object-contain"
                />
              )}
            </div>
          )}
          
          <div className="mt-2 flex items-center text-sm text-muted-foreground">
            <Check className="h-4 w-4 mr-1 text-green-500" />
            <span>上传成功</span>
          </div>
        </div>
      )}
    </div>
  );
} 