import { NextRequest, NextResponse } from "next/server";
import { cookies } from "next/headers";
import { createAdminClient } from "@/utils/supabase/admin";
import { createClient } from "@/utils/supabase/server";

// 文件大小限制：10MB
const MAX_FILE_SIZE = 10 * 1024 * 1024;

export async function POST(request: NextRequest) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 处理表单数据
    const formData = await request.formData();
    const file = formData.get("file") as File;
    const bucket = formData.get("bucket") as string;
    
    if (!file) {
      return NextResponse.json(
        { error: "未提供文件" },
        { status: 400 }
      );
    }
    
    if (!bucket) {
      return NextResponse.json(
        { error: "未指定存储桶" },
        { status: 400 }
      );
    }
    
    // 检查文件大小
    if (file.size > MAX_FILE_SIZE) {
      return NextResponse.json(
        { error: "文件大小超过限制（10MB）" },
        { status: 400 }
      );
    }
    
    // 生成唯一文件名
    const timestamp = new Date().getTime();
    const fileExtension = file.name.split(".").pop();
    const fileName = `${timestamp}-${Math.random().toString(36).substring(2, 15)}.${fileExtension}`;
    
    // 上传到Supabase存储
    const { data, error } = await adminClient
      .storage
      .from(bucket)
      .upload(fileName, file, {
        contentType: file.type,
        upsert: false,
      });
      
    if (error) {
      console.error("文件上传错误:", error);
      return NextResponse.json(
        { error: error.message },
        { status: 500 }
      );
    }
    
    // 获取公共URL
    const { data: { publicUrl } } = adminClient
      .storage
      .from(bucket)
      .getPublicUrl(data.path);
      
    return NextResponse.json({
      url: publicUrl,
      path: data.path,
    });
    
  } catch (error) {
    console.error("上传处理错误:", error);
    return NextResponse.json(
      { error: "文件上传处理失败" },
      { status: 500 }
    );
  }
} 