import { NextRequest, NextResponse } from "next/server";
import { createAdminClient } from "@/utils/supabase/admin";
import { createClient } from "@/utils/supabase/server";

export async function POST(request: NextRequest) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取请求体
    const toolData = await request.json();
    
    // 验证必填字段
    if (!toolData.name || !toolData.slug || !toolData.description || 
        !toolData.website_url || !toolData.category_id) {
      return NextResponse.json(
        { error: "缺少必填字段" },
        { status: 400 }
      );
    }
    
    // 验证slug格式
    const slugRegex = /^[a-z0-9-]+$/;
    if (!slugRegex.test(toolData.slug)) {
      return NextResponse.json(
        { error: "URL短识只能包含小写字母、数字和连字符" },
        { status: 400 }
      );
    }
    
    // 验证slug唯一性
    const { data: existingTool } = await adminClient
      .from("ai_tools")
      .select("id")
      .eq("slug", toolData.slug)
      .single();
      
    if (existingTool) {
      return NextResponse.json(
        { error: "该URL短识已被使用，请选择其他标识" },
        { status: 400 }
      );
    }
    
    // 添加创建时间和更新时间
    const now = new Date().toISOString();
    const toolWithTimestamps = {
      ...toolData,
      created_at: now,
      updated_at: now,
      user_id: user.id
    };
    
    // 插入数据
    const { data, error } = await adminClient
      .from("ai_tools")
      .insert(toolWithTimestamps)
      .select()
      .single();
      
    if (error) {
      console.error("添加工具错误:", error);
      return NextResponse.json(
        { error: "添加工具失败" },
        { status: 500 }
      );
    }
    
    return NextResponse.json({ 
      message: "工具添加成功", 
      tool: data 
    });
    
  } catch (error) {
    console.error("处理添加工具请求错误:", error);
    return NextResponse.json(
      { error: "处理添加工具请求失败" },
      { status: 500 }
    );
  }
}

// 获取所有工具列表
export async function GET(request: NextRequest) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取查询参数
    const url = new URL(request.url);
    const page = parseInt(url.searchParams.get("page") || "1");
    const pageSize = parseInt(url.searchParams.get("pageSize") || "10");
    
    // 计算分页偏移量
    const from = (page - 1) * pageSize;
    const to = from + pageSize - 1;
    
    // 查询工具总数
    const { count } = await adminClient
      .from("ai_tools")
      .select("*", { count: "exact", head: true });
      
    // 查询工具列表
    const { data: tools, error } = await adminClient
      .from("ai_tools")
      .select(`
        *,
        category:categories(name)
      `)
      .order("created_at", { ascending: false })
      .range(from, to);
      
    if (error) {
      console.error("获取工具列表错误:", error);
      return NextResponse.json(
        { error: "获取工具列表失败" },
        { status: 500 }
      );
    }
    
    return NextResponse.json({
      tools,
      pagination: {
        total: count,
        page,
        pageSize,
        pageCount: Math.ceil((count || 0) / pageSize)
      }
    });
    
  } catch (error) {
    console.error("处理获取工具列表请求错误:", error);
    return NextResponse.json(
      { error: "处理获取工具列表请求失败" },
      { status: 500 }
    );
  }
} 