import { NextRequest, NextResponse } from "next/server";
import { createAdminClient } from "@/utils/supabase/admin";
import { createClient } from "@/utils/supabase/server";

// 获取所有用户提交的工具
export async function GET(request: NextRequest) {
  try {
    // 验证管理员权限
    const adminClient = createAdminClient();
    const supabase = await createClient();
    
    const { data: { user } } = await supabase.auth.getUser();
    if (!user) {
      return NextResponse.json(
        { error: "未授权，请先登录" },
        { status: 401 }
      );
    }
    
    const { data: adminUser } = await adminClient
      .from("admin_users")
      .select("*")
      .eq("user_id", user.id)
      .single();
      
    if (!adminUser) {
      return NextResponse.json(
        { error: "您没有管理员权限" },
        { status: 403 }
      );
    }
    
    // 获取查询参数
    const url = new URL(request.url);
    const page = parseInt(url.searchParams.get("page") || "1");
    const pageSize = parseInt(url.searchParams.get("pageSize") || "10");
    const status = url.searchParams.get("status") || "pending";
    
    // 计算分页偏移量
    const from = (page - 1) * pageSize;
    const to = from + pageSize - 1;
    
    // 查询总数
    const { count } = await adminClient
      .from("tool_submissions")
      .select("*", { count: "exact", head: true })
      .eq("status", status);
      
    // 查询提交列表
    const { data: submissions, error } = await adminClient
      .from("tool_submissions")
      .select(`
        *,
        category:categories(id, name)
      `)
      .eq("status", status)
      .order("created_at", { ascending: false })
      .range(from, to);
      
    if (error) {
      console.error("获取提交列表错误:", error);
      return NextResponse.json(
        { error: "获取提交列表失败" },
        { status: 500 }
      );
    }
    
    // 如果需要用户信息，可以单独查询
    if (submissions && submissions.length > 0) {
      // 提取所有不重复的用户ID
      const userIds = Array.from(new Set(submissions.map(sub => sub.user_id)));
      
      // 查询这些用户的信息
      const { data: users } = await adminClient
        .from("auth.users")
        .select("id, email")
        .in("id", userIds);
        
      // 如果获取到用户信息，将其添加到提交数据中
      if (users) {
        const userMap = new Map(users.map(user => [user.id, user]));
        
        for (const submission of submissions) {
          const userInfo = userMap.get(submission.user_id);
          if (userInfo) {
            submission.user = {
              id: userInfo.id,
              email: userInfo.email
            };
          }
        }
      }
    }
    
    return NextResponse.json({
      submissions,
      pagination: {
        total: count,
        page,
        pageSize,
        pageCount: Math.ceil((count || 0) / pageSize)
      }
    });
    
  } catch (error) {
    console.error("处理获取提交列表请求错误:", error);
    return NextResponse.json(
      { error: "处理获取提交列表请求失败" },
      { status: 500 }
    );
  }
} 