"use client";

import { useState, useEffect } from "react";
import Link from "next/link";
import { useRouter } from "next/navigation";
import { useToast } from "@/components/ui/use-toast";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogClose,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog";
import { PlusCircle, Pencil, Eye, Trash2, ExternalLink, Loader2 } from "lucide-react";

interface Tool {
  id: string;
  name: string;
  slug: string;
  description: string;
  full_description?: string;
  website_url: string;
  logo_url: string | null;
  image_url?: string | null;
  type?: string;
  category: {
    name: string;
  };
  created_at: string;
}

interface PaginationData {
  total: number;
  page: number;
  pageSize: number;
  pageCount: number;
}

export default function ToolsPage() {
  const router = useRouter();
  const { toast } = useToast();
  const [tools, setTools] = useState<Tool[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isDeleting, setIsDeleting] = useState(false);
  const [pagination, setPagination] = useState<PaginationData>({
    total: 0,
    page: 1,
    pageSize: 10,
    pageCount: 0
  });
  
  // 用于查看详情和删除确认的状态
  const [viewingTool, setViewingTool] = useState<Tool | null>(null);
  const [deletingToolId, setDeletingToolId] = useState<string | null>(null);

  // 获取工具列表
  const fetchTools = async () => {
    try {
      setIsLoading(true);
      const response = await fetch(`/api/admin/tools?page=${pagination.page}&pageSize=${pagination.pageSize}`);
      
      if (!response.ok) {
        throw new Error("获取工具列表失败");
      }
      
      const data = await response.json();
      setTools(data.tools || []);
      setPagination(data.pagination || pagination);
    } catch (error) {
      console.error("加载工具列表失败:", error);
      toast({
        title: "错误",
        description: "无法加载工具列表",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };
  
  useEffect(() => {
    fetchTools();
  }, [pagination.page, pagination.pageSize, toast]);

  // 处理页面变化
  const handlePageChange = (page: number) => {
    setPagination((prev) => ({ ...prev, page }));
  };

  // 查看工具详情
  const handleViewTool = async (tool: Tool) => {
    try {
      const response = await fetch(`/api/admin/tools/${tool.id}`);
      
      if (!response.ok) {
        throw new Error("获取工具详情失败");
      }
      
      const data = await response.json();
      setViewingTool(data.tool);
    } catch (error) {
      console.error("加载工具详情失败:", error);
      toast({
        title: "错误",
        description: "无法加载工具详情",
        variant: "destructive",
      });
    }
  };

  // 删除工具
  const handleDeleteTool = async () => {
    if (!deletingToolId) return;
    
    setIsDeleting(true);
    
    try {
      const response = await fetch(`/api/admin/tools/${deletingToolId}`, {
        method: "DELETE",
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "删除工具失败");
      }
      
      toast({
        title: "成功",
        description: "工具已成功删除",
      });
      
      // 重新加载工具列表
      fetchTools();
      
    } catch (error) {
      console.error("删除工具失败:", error);
      toast({
        title: "错误",
        description: error instanceof Error ? error.message : "删除工具失败",
        variant: "destructive",
      });
    } finally {
      setIsDeleting(false);
      setDeletingToolId(null);
    }
  };

  // 格式化日期
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("zh-CN", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold">AI 工具管理</h1>
        <Link href="/admin/tools/new">
          <Button>
            <PlusCircle className="h-4 w-4 mr-2" />
            添加工具
          </Button>
        </Link>
      </div>

      {isLoading ? (
        <div className="h-60 flex items-center justify-center">
          <p className="text-muted-foreground">加载中...</p>
        </div>
      ) : tools.length === 0 ? (
        <div className="h-60 flex flex-col items-center justify-center space-y-4">
          <p className="text-muted-foreground text-lg">暂无工具数据</p>
          <Link href="/admin/tools/new">
            <Button>
              <PlusCircle className="h-4 w-4 mr-2" />
              添加第一个工具
            </Button>
          </Link>
        </div>
      ) : (
        <>
          <div className="border rounded-md">
            <table className="w-full">
              <thead>
                <tr className="border-b">
                  <th className="p-3 text-left font-medium">名称</th>
                  <th className="p-3 text-left font-medium">URL短识</th>
                  <th className="p-3 text-left font-medium">分类</th>
                  <th className="p-3 text-left font-medium">添加时间</th>
                  <th className="p-3 text-right font-medium">操作</th>
                </tr>
              </thead>
              <tbody>
                {tools.map((tool) => (
                  <tr key={tool.id} className="border-b last:border-b-0">
                    <td className="p-3">
                      <div className="flex items-center space-x-3">
                        {tool.logo_url && (
                          <div className="h-8 w-8 rounded-md overflow-hidden bg-muted">
                            <img
                              src={tool.logo_url}
                              alt={tool.name}
                              className="h-full w-full object-cover"
                            />
                          </div>
                        )}
                        <span className="font-medium">{tool.name}</span>
                      </div>
                    </td>
                    <td className="p-3 text-sm">{tool.slug}</td>
                    <td className="p-3">{tool.category?.name || "-"}</td>
                    <td className="p-3 text-sm">{formatDate(tool.created_at)}</td>
                    <td className="p-3 text-right">
                      <div className="flex items-center justify-end space-x-2">
                        <Button 
                          variant="outline" 
                          size="icon"
                          onClick={() => handleViewTool(tool)}
                          title="查看详情"
                        >
                          <Eye className="h-4 w-4" />
                        </Button>
                        <Link href={`/admin/tools/${tool.id}`}>
                          <Button 
                            variant="outline" 
                            size="icon"
                            title="编辑"
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                        </Link>
                        <Button 
                          variant="outline" 
                          size="icon"
                          onClick={() => setDeletingToolId(tool.id)}
                          title="删除"
                          className="text-destructive"
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>

          {/* 分页控件 */}
          {pagination.pageCount > 1 && (
            <div className="flex items-center justify-center space-x-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => handlePageChange(pagination.page - 1)}
                disabled={pagination.page === 1}
              >
                上一页
              </Button>
              
              {Array.from({ length: pagination.pageCount }, (_, i) => i + 1).map((page) => (
                <Button
                  key={page}
                  variant={page === pagination.page ? "default" : "outline"}
                  size="sm"
                  onClick={() => handlePageChange(page)}
                >
                  {page}
                </Button>
              ))}
              
              <Button
                variant="outline"
                size="sm"
                onClick={() => handlePageChange(pagination.page + 1)}
                disabled={pagination.page === pagination.pageCount}
              >
                下一页
              </Button>
            </div>
          )}

          {/* 工具详情对话框 */}
          <Dialog open={viewingTool !== null} onOpenChange={(isOpen: boolean) => !isOpen && setViewingTool(null)}>
            <DialogContent className="max-w-3xl max-h-[80vh] overflow-auto">
              {viewingTool && (
                <>
                  <DialogHeader>
                    <DialogTitle className="flex items-center gap-3">
                      {viewingTool.logo_url && (
                        <div className="h-8 w-8 rounded-md overflow-hidden bg-muted">
                          <img
                            src={viewingTool.logo_url}
                            alt={viewingTool.name}
                            className="h-full w-full object-cover"
                          />
                        </div>
                      )}
                      {viewingTool.name}
                    </DialogTitle>
                    <DialogDescription>
                      <a 
                        href={viewingTool.website_url} 
                        target="_blank" 
                        rel="noopener noreferrer"
                        className="flex items-center text-blue-500 hover:underline mt-2"
                      >
                        {viewingTool.website_url}
                        <ExternalLink className="h-3 w-3 ml-1" />
                      </a>
                    </DialogDescription>
                  </DialogHeader>

                  <div className="space-y-4 my-4">
                    <div>
                      <h3 className="text-sm font-semibold text-muted-foreground">分类</h3>
                      <p>{viewingTool.category?.name}</p>
                    </div>
                    
                    {viewingTool.type && (
                      <div>
                        <h3 className="text-sm font-semibold text-muted-foreground">类型</h3>
                        <p>{viewingTool.type}</p>
                      </div>
                    )}
                    
                    <div>
                      <h3 className="text-sm font-semibold text-muted-foreground">描述</h3>
                      <p>{viewingTool.description}</p>
                    </div>
                    
                    {viewingTool.full_description && (
                      <div>
                        <h3 className="text-sm font-semibold text-muted-foreground">详细介绍</h3>
                        <div className="mt-2 border rounded-md p-4 bg-muted/30">
                          <pre className="whitespace-pre-wrap text-sm">{viewingTool.full_description}</pre>
                        </div>
                      </div>
                    )}
                    
                    {viewingTool.image_url && (
                      <div>
                        <h3 className="text-sm font-semibold text-muted-foreground">预览图</h3>
                        <div className="mt-2 rounded-md overflow-hidden border">
                          <img
                            src={viewingTool.image_url}
                            alt={`${viewingTool.name} 预览`}
                            className="max-h-64 w-auto object-contain mx-auto"
                          />
                        </div>
                      </div>
                    )}
                  </div>

                  <DialogFooter>
                    <DialogClose asChild>
                      <Button variant="outline">关闭</Button>
                    </DialogClose>
                    <Link href={`/admin/tools/${viewingTool.id}`}>
                      <Button variant="default">
                        <Pencil className="h-4 w-4 mr-2" />
                        编辑
                      </Button>
                    </Link>
                  </DialogFooter>
                </>
              )}
            </DialogContent>
          </Dialog>

          {/* 删除确认对话框 */}
          <AlertDialog open={deletingToolId !== null} onOpenChange={(isOpen: boolean) => !isOpen && setDeletingToolId(null)}>
            <AlertDialogContent>
              <AlertDialogHeader>
                <AlertDialogTitle>确认删除</AlertDialogTitle>
                <AlertDialogDescription>
                  您确定要删除此AI工具吗？此操作不可逆，删除后数据将无法恢复。
                </AlertDialogDescription>
              </AlertDialogHeader>
              <AlertDialogFooter>
                <AlertDialogCancel disabled={isDeleting}>取消</AlertDialogCancel>
                <AlertDialogAction 
                  onClick={(e: React.MouseEvent) => {
                    e.preventDefault();
                    handleDeleteTool();
                  }}
                  disabled={isDeleting}
                  className="bg-destructive hover:bg-destructive/90"
                >
                  {isDeleting ? (
                    <>
                      <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                      删除中...
                    </>
                  ) : (
                    <>
                      <Trash2 className="h-4 w-4 mr-2" />
                      确认删除
                    </>
                  )}
                </AlertDialogAction>
              </AlertDialogFooter>
            </AlertDialogContent>
          </AlertDialog>
        </>
      )}
    </div>
  );
} 