"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import { useToast } from "@/components/ui/use-toast";
import { Button } from "@/components/ui/button";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
  DialogClose,
} from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
  AlertDialogTrigger,
} from "@/components/ui/alert-dialog";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { 
  Card, 
  CardContent, 
  CardDescription, 
  CardFooter, 
  CardHeader, 
  CardTitle 
} from "@/components/ui/card";
import { 
  Tabs, 
  TabsContent, 
  TabsList, 
  TabsTrigger 
} from "@/components/ui/tabs";
import { Textarea } from "@/components/ui/textarea";
import { CheckCircle, Clock, XCircle, ExternalLink, EyeIcon, ThumbsUp, ThumbsDown, Loader2 } from "lucide-react";
import Link from "next/link";
import { formatDistanceToNow } from "date-fns";
import { zhCN } from "date-fns/locale";

interface Submission {
  id: string;
  name: string;
  slug: string;
  description: string;
  full_description?: string;
  website_url: string;
  logo_url?: string;
  image_url?: string;
  category_id: string;
  user_id: string;
  status: "pending" | "approved" | "rejected";
  admin_comments?: string;
  created_at: string;
  updated_at: string;
  processed_at?: string;
  ai_tool_id?: string;
  category?: {
    id: string;
    name: string;
  };
  user?: {
    id: string;
    email: string;
  };
}

interface PaginationData {
  total: number;
  page: number;
  pageSize: number;
  pageCount: number;
}

export default function SubmissionsPage() {
  const router = useRouter();
  const { toast } = useToast();
  const [submissions, setSubmissions] = useState<Submission[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isProcessing, setIsProcessing] = useState(false);
  const [currentStatus, setCurrentStatus] = useState<string>("pending");
  const [pagination, setPagination] = useState<PaginationData>({
    total: 0,
    page: 1,
    pageSize: 10,
    pageCount: 0
  });
  
  // 用于查看详情的状态
  const [viewingSubmission, setViewingSubmission] = useState<Submission | null>(null);
  const [rejectingSubmission, setRejectingSubmission] = useState<Submission | null>(null);
  const [rejectReason, setRejectReason] = useState("");
  
  // 获取提交列表
  const fetchSubmissions = async () => {
    try {
      setIsLoading(true);
      const response = await fetch(
        `/api/admin/submissions?page=${pagination.page}&pageSize=${pagination.pageSize}&status=${currentStatus}`
      );
      
      if (!response.ok) {
        throw new Error("获取提交列表失败");
      }
      
      const data = await response.json();
      setSubmissions(data.submissions || []);
      setPagination(data.pagination || pagination);
    } catch (error) {
      console.error("加载提交列表失败:", error);
      toast({
        title: "错误",
        description: "无法加载提交列表",
        variant: "destructive",
      });
    } finally {
      setIsLoading(false);
    }
  };
  
  useEffect(() => {
    fetchSubmissions();
  }, [pagination.page, pagination.pageSize, currentStatus, toast]);

  // 处理页面变化
  const handlePageChange = (page: number) => {
    setPagination((prev) => ({ ...prev, page }));
  };

  // 查看提交详情
  const handleViewSubmission = async (submission: Submission) => {
    try {
      const response = await fetch(`/api/admin/submissions/${submission.id}`);
      
      if (!response.ok) {
        throw new Error("获取提交详情失败");
      }
      
      const data = await response.json();
      setViewingSubmission(data.submission);
    } catch (error) {
      console.error("加载提交详情失败:", error);
      toast({
        title: "错误",
        description: "无法加载提交详情",
        variant: "destructive",
      });
    }
  };

  // 处理通过提交
  const handleApproveSubmission = async (submission: Submission) => {
    if (!submission) return;
    
    setIsProcessing(true);
    
    try {
      const response = await fetch(`/api/admin/submissions/${submission.id}`, {
        method: "POST"
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "通过提交失败");
      }
      
      toast({
        title: "成功",
        description: "已通过提交，AI工具已添加",
      });
      
      // 关闭弹窗并重新加载列表
      setViewingSubmission(null);
      fetchSubmissions();
      
    } catch (error) {
      console.error("通过提交失败:", error);
      toast({
        title: "错误",
        description: error instanceof Error ? error.message : "通过提交失败",
        variant: "destructive",
      });
    } finally {
      setIsProcessing(false);
    }
  };

  // 处理驳回提交
  const handleRejectSubmission = async () => {
    if (!rejectingSubmission) return;
    
    setIsProcessing(true);
    
    try {
      const response = await fetch(`/api/admin/submissions/${rejectingSubmission.id}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          admin_comments: rejectReason
        }),
      });
      
      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "驳回提交失败");
      }
      
      toast({
        title: "成功",
        description: "已驳回提交",
      });
      
      // 关闭弹窗并重新加载列表
      setRejectingSubmission(null);
      setRejectReason("");
      setViewingSubmission(null);
      fetchSubmissions();
      
    } catch (error) {
      console.error("驳回提交失败:", error);
      toast({
        title: "错误",
        description: error instanceof Error ? error.message : "驳回提交失败",
        variant: "destructive",
      });
    } finally {
      setIsProcessing(false);
    }
  };

  // 格式化相对时间
  const formatRelativeTime = (dateString: string) => {
    return formatDistanceToNow(new Date(dateString), {
      addSuffix: true,
      locale: zhCN
    });
  };

  const getStatusBadge = (status: string) => {
    switch (status) {
      case "pending":
        return (
          <div className="flex items-center text-amber-500">
            <Clock className="h-4 w-4 mr-1" />
            <span>待审核</span>
          </div>
        );
      case "approved":
        return (
          <div className="flex items-center text-green-500">
            <CheckCircle className="h-4 w-4 mr-1" />
            <span>已通过</span>
          </div>
        );
      case "rejected":
        return (
          <div className="flex items-center text-red-500">
            <XCircle className="h-4 w-4 mr-1" />
            <span>已驳回</span>
          </div>
        );
      default:
        return <span>{status}</span>;
    }
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold">审核中心</h1>
      </div>

      <Tabs defaultValue="pending" value={currentStatus} onValueChange={setCurrentStatus}>
        <TabsList>
          <TabsTrigger value="pending">待审核</TabsTrigger>
          <TabsTrigger value="approved">已通过</TabsTrigger>
          <TabsTrigger value="rejected">已驳回</TabsTrigger>
        </TabsList>
      </Tabs>

      {isLoading ? (
        <div className="h-60 flex items-center justify-center">
          <p className="text-muted-foreground">加载中...</p>
        </div>
      ) : submissions.length === 0 ? (
        <div className="h-60 flex flex-col items-center justify-center space-y-4">
          <p className="text-muted-foreground text-lg">暂无{currentStatus === "pending" ? "待审核" : currentStatus === "approved" ? "已通过" : "已驳回"}的提交</p>
        </div>
      ) : (
        <>
          <div className="grid grid-cols-1 gap-4 md:grid-cols-2 lg:grid-cols-3">
            {submissions.map((submission) => (
              <Card key={submission.id} className="overflow-hidden">
                <CardHeader className="pb-2">
                  <CardTitle className="flex items-center gap-2 truncate">
                    {submission.logo_url && (
                      <div className="h-6 w-6 rounded-md overflow-hidden flex-shrink-0 bg-muted">
                        <img
                          src={submission.logo_url}
                          alt={submission.name}
                          className="h-full w-full object-cover"
                        />
                      </div>
                    )}
                    <span className="truncate">{submission.name}</span>
                  </CardTitle>
                  <CardDescription className="flex items-center justify-between">
                    <span>{submission.category?.name || "-"}</span>
                    {getStatusBadge(submission.status)}
                  </CardDescription>
                </CardHeader>
                <CardContent className="pb-2">
                  <p className="text-sm line-clamp-3">{submission.description}</p>
                </CardContent>
                <CardFooter className="pt-1 flex flex-col items-start">
                  <div className="w-full flex items-center justify-between text-xs text-muted-foreground mb-2">
                    <span>提交于 {formatRelativeTime(submission.created_at)}</span>
                    {submission.user?.email && (
                      <span title={submission.user.email}>
                        {submission.user.email.length > 15 
                          ? `${submission.user.email.substring(0, 15)}...` 
                          : submission.user.email}
                      </span>
                    )}
                  </div>
                  <Button 
                    variant="outline" 
                    size="sm"
                    className="w-full"
                    onClick={() => handleViewSubmission(submission)}
                  >
                    <EyeIcon className="h-4 w-4 mr-2" />
                    查看详情
                  </Button>
                </CardFooter>
              </Card>
            ))}
          </div>

          {/* 分页控件 */}
          {pagination.pageCount > 1 && (
            <div className="flex items-center justify-center space-x-2">
              <Button
                variant="outline"
                size="sm"
                onClick={() => handlePageChange(pagination.page - 1)}
                disabled={pagination.page === 1}
              >
                上一页
              </Button>
              
              {Array.from({ length: pagination.pageCount }, (_, i) => i + 1).map((page) => (
                <Button
                  key={page}
                  variant={page === pagination.page ? "default" : "outline"}
                  size="sm"
                  onClick={() => handlePageChange(page)}
                >
                  {page}
                </Button>
              ))}
              
              <Button
                variant="outline"
                size="sm"
                onClick={() => handlePageChange(pagination.page + 1)}
                disabled={pagination.page === pagination.pageCount}
              >
                下一页
              </Button>
            </div>
          )}

          {/* 提交详情对话框 */}
          <Dialog open={viewingSubmission !== null} onOpenChange={(isOpen: boolean) => !isOpen && setViewingSubmission(null)}>
            <DialogContent className="max-w-3xl max-h-[80vh] overflow-auto">
              {viewingSubmission && (
                <>
                  <DialogHeader>
                    <DialogTitle className="flex items-center gap-3">
                      {viewingSubmission.logo_url && (
                        <div className="h-8 w-8 rounded-md overflow-hidden bg-muted">
                          <img
                            src={viewingSubmission.logo_url}
                            alt={viewingSubmission.name}
                            className="h-full w-full object-cover"
                          />
                        </div>
                      )}
                      {viewingSubmission.name}
                    </DialogTitle>
                    <DialogDescription>
                      <a 
                        href={viewingSubmission.website_url} 
                        target="_blank" 
                        rel="noopener noreferrer"
                        className="flex items-center text-blue-500 hover:underline mt-2"
                      >
                        {viewingSubmission.website_url}
                        <ExternalLink className="h-3 w-3 ml-1" />
                      </a>
                    </DialogDescription>
                  </DialogHeader>

                  <div className="space-y-4 my-4">
                    <div>
                      <h3 className="text-sm font-semibold text-muted-foreground">分类</h3>
                      <p>{viewingSubmission.category?.name}</p>
                    </div>
                    
                    <div>
                      <h3 className="text-sm font-semibold text-muted-foreground">URL短识</h3>
                      <p>{viewingSubmission.slug}</p>
                    </div>
                    
                    <div>
                      <h3 className="text-sm font-semibold text-muted-foreground">描述</h3>
                      <p>{viewingSubmission.description}</p>
                    </div>
                    
                    {viewingSubmission.full_description && (
                      <div>
                        <h3 className="text-sm font-semibold text-muted-foreground">详细介绍</h3>
                        <div className="mt-2 border rounded-md p-4 bg-muted/30">
                          <pre className="whitespace-pre-wrap text-sm">{viewingSubmission.full_description}</pre>
                        </div>
                      </div>
                    )}
                    
                    {viewingSubmission.image_url && (
                      <div>
                        <h3 className="text-sm font-semibold text-muted-foreground">预览图</h3>
                        <div className="mt-2 rounded-md overflow-hidden border">
                          <img
                            src={viewingSubmission.image_url}
                            alt={`${viewingSubmission.name} 预览`}
                            className="max-h-64 w-auto object-contain mx-auto"
                          />
                        </div>
                      </div>
                    )}
                    
                    <div>
                      <h3 className="text-sm font-semibold text-muted-foreground">提交信息</h3>
                      <div className="flex flex-col gap-1 mt-1 text-sm">
                        <p>
                          <span className="text-muted-foreground">提交时间: </span>
                          {new Date(viewingSubmission.created_at).toLocaleString()}
                        </p>
                        <p>
                          <span className="text-muted-foreground">提交用户: </span>
                          {viewingSubmission.user?.email || "-"}
                        </p>
                        {viewingSubmission.status !== "pending" && (
                          <>
                            <p>
                              <span className="text-muted-foreground">处理时间: </span>
                              {viewingSubmission.processed_at ? new Date(viewingSubmission.processed_at).toLocaleString() : "-"}
                            </p>
                            {viewingSubmission.status === "rejected" && viewingSubmission.admin_comments && (
                              <p>
                                <span className="text-muted-foreground">驳回原因: </span>
                                {viewingSubmission.admin_comments}
                              </p>
                            )}
                          </>
                        )}
                      </div>
                    </div>
                  </div>

                  <DialogFooter>
                    {viewingSubmission.status === "pending" ? (
                      <>
                        <Button
                          type="button"
                          variant="outline"
                          className="gap-2 text-red-500 hover:text-red-500" 
                          onClick={() => {
                            setRejectingSubmission(viewingSubmission);
                          }}
                          disabled={isProcessing}
                        >
                          <ThumbsDown className="h-4 w-4" />
                          驳回
                        </Button>
                        <Button
                          type="button"
                          variant="default"
                          className="gap-2"
                          onClick={() => handleApproveSubmission(viewingSubmission)}
                          disabled={isProcessing}
                        >
                          {isProcessing ? (
                            <>
                              <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                              处理中...
                            </>
                          ) : (
                            <>
                              <ThumbsUp className="h-4 w-4" />
                              通过
                            </>
                          )}
                        </Button>
                      </>
                    ) : (
                      <DialogClose asChild>
                        <Button variant="outline">关闭</Button>
                      </DialogClose>
                    )}
                  </DialogFooter>
                </>
              )}
            </DialogContent>
          </Dialog>

          {/* 驳回原因对话框 */}
          <AlertDialog open={rejectingSubmission !== null} onOpenChange={(isOpen) => !isOpen && setRejectingSubmission(null)}>
            <AlertDialogContent>
              <AlertDialogHeader>
                <AlertDialogTitle>驳回提交</AlertDialogTitle>
                <AlertDialogDescription>
                  请填写驳回原因，该原因将会发送给用户以帮助他们理解为何提交未被通过。
                </AlertDialogDescription>
                <Textarea
                  placeholder="请输入驳回原因..."
                  className="mt-4"
                  value={rejectReason}
                  onChange={(e) => setRejectReason(e.target.value)}
                />
              </AlertDialogHeader>
              <AlertDialogFooter>
                <AlertDialogCancel disabled={isProcessing}>取消</AlertDialogCancel>
                <AlertDialogAction 
                  onClick={(e: React.MouseEvent) => {
                    e.preventDefault();
                    handleRejectSubmission();
                  }}
                  disabled={isProcessing}
                  className="bg-destructive hover:bg-destructive/90"
                >
                  {isProcessing ? (
                    <>
                      <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                      处理中...
                    </>
                  ) : "确认驳回"}
                </AlertDialogAction>
              </AlertDialogFooter>
            </AlertDialogContent>
          </AlertDialog>
        </>
      )}
    </div>
  );
} 