import { createAdminClient } from "@/utils/supabase/admin";
import { BarChart, Users, FileCheck, ArrowRight, Plus } from "lucide-react";
import Link from "next/link";

// 统计卡片组件
function StatCard({ 
  title, 
  value, 
  icon: Icon,
  className = "bg-card" 
}: { 
  title: string; 
  value: number;
  icon: React.ElementType;
  className?: string;
}) {
  return (
    <div className={`p-6 rounded-lg border shadow-sm ${className}`}>
      <div className="flex justify-between items-start">
        <div>
          <p className="text-sm font-medium text-muted-foreground">{title}</p>
          <h3 className="text-3xl font-bold mt-2">{value}</h3>
        </div>
        <div className="bg-primary/10 p-3 rounded-full">
          <Icon className="h-6 w-6 text-primary" />
        </div>
      </div>
    </div>
  );
}

export default async function AdminDashboard() {
  // 使用 adminClient 绕过 RLS 策略
  const adminClient = createAdminClient();
  
  // 获取今天的日期范围
  const now = new Date();
  const today = new Date(now.getFullYear(), now.getMonth(), now.getDate()).toISOString();
  const tomorrow = new Date(now.getFullYear(), now.getMonth(), now.getDate() + 1).toISOString();
  
  // 1. 获取 AI 工具总数
  const { count: toolsCount, error: toolsError } = await adminClient
    .from('ai_tools')
    .select('*', { count: 'exact', head: true });
    
  if (toolsError) {
    console.error("获取AI工具总数失败:", toolsError);
  }
  
  // 2. 获取分类总数
  const { count: categoriesCount, error: categoriesError } = await adminClient
    .from('categories')
    .select('*', { count: 'exact', head: true });
    
  if (categoriesError) {
    console.error("获取分类总数失败:", categoriesError);
  }
  
  // 3. 获取待审核提交数
  const { count: pendingSubmissionsCount, error: pendingError } = await adminClient
    .from('tool_submissions')
    .select('*', { count: 'exact', head: true })
    .eq('status', 'pending');
    
  if (pendingError) {
    console.error("获取待审核提交数失败:", pendingError);
  }
  
  // 4. 获取今日提交数 - 使用时间范围查询
  const { count: todaySubmissionsCount, error: todayError } = await adminClient
    .from('tool_submissions')
    .select('*', { count: 'exact', head: true })
    .gte('created_at', today)
    .lt('created_at', tomorrow);
    
  if (todayError) {
    console.error("获取今日提交数失败:", todayError);
  }
    
  return (
    <div className="space-y-6">
      <div className="flex justify-between items-center">
        <h1 className="text-2xl font-bold">管理后台概览</h1>
        <Link 
          href="/admin/tools/new"
          className="flex items-center gap-2 bg-primary text-primary-foreground px-4 py-2 rounded-md text-sm font-medium"
        >
          <Plus className="h-4 w-4" />
          添加 AI 工具
        </Link>
      </div>
      
      {/* 统计卡片 */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <StatCard 
          title="AI 工具总数" 
          value={toolsCount || 0} 
          icon={BarChart} 
        />
        <StatCard 
          title="分类总数" 
          value={categoriesCount || 0} 
          icon={FileCheck} 
        />
        <StatCard 
          title="今日提交" 
          value={todaySubmissionsCount || 0} 
          icon={BarChart} 
          className="bg-card border-blue-100 dark:border-blue-900"
        />
        <StatCard 
          title="待审核" 
          value={pendingSubmissionsCount || 0} 
          icon={Users} 
          className="bg-card border-amber-100 dark:border-amber-900"
        />
      </div>
      
      {/* 快捷操作区 */}
      <div className="mt-8">
        <h2 className="text-lg font-medium mb-4">快捷操作</h2>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <Link 
            href="/admin/tools"
            className="p-4 border rounded-lg flex justify-between items-center hover:bg-accent transition-colors"
          >
            <div>
              <h3 className="font-medium">管理 AI 工具</h3>
              <p className="text-sm text-muted-foreground mt-1">查看和编辑已上架的 AI 工具</p>
            </div>
            <ArrowRight className="h-5 w-5 text-muted-foreground" />
          </Link>
          
          <Link 
            href="/admin/submissions"
            className="p-4 border rounded-lg flex justify-between items-center hover:bg-accent transition-colors"
          >
            <div>
              <h3 className="font-medium">审核工具提交</h3>
              <p className="text-sm text-muted-foreground mt-1">处理用户提交的 AI 工具申请</p>
            </div>
            <ArrowRight className="h-5 w-5 text-muted-foreground" />
          </Link>
          
          <Link 
            href="/admin/users"
            className="p-4 border rounded-lg flex justify-between items-center hover:bg-accent transition-colors"
          >
            <div>
              <h3 className="font-medium">用户管理</h3>
              <p className="text-sm text-muted-foreground mt-1">管理用户权限和账户</p>
            </div>
            <ArrowRight className="h-5 w-5 text-muted-foreground" />
          </Link>
        </div>
      </div>
    </div>
  );
} 