"use server";

import { encodedRedirect } from "@/utils/utils";
import { createClient } from "@/utils/supabase/server";
import { createAdminClient } from "@/utils/supabase/admin";
import { headers } from "next/headers";
import { redirect } from "next/navigation";

export const signUpAction = async (formData: FormData) => {
  const email = formData.get("email")?.toString();
  const password = formData.get("password")?.toString();
  const supabase = await createClient();
  const origin = (await headers()).get("origin");

  if (!email || !password) {
    return encodedRedirect(
      "error",
      "/sign-up",
      "请填写电子邮箱和密码",
    );
  }

  const { error } = await supabase.auth.signUp({
    email,
    password,
    options: {
      emailRedirectTo: `${origin}/auth/callback`,
    },
  });

  if (error) {
    console.error(error.code + " " + error.message);
    return encodedRedirect("error", "/sign-up", error.message);
  } else {
    return encodedRedirect(
      "success",
      "/sign-up",
      "注册成功！请检查您的邮箱以验证账户。",
    );
  }
};

export const signInAction = async (formData: FormData) => {
  const email = formData.get("email") as string;
  const password = formData.get("password") as string;
  const supabase = await createClient();

  // 验证用户名和密码
  const { data, error } = await supabase.auth.signInWithPassword({
    email,
    password,
  });

  if (error) {
    console.error("登录失败:", error.message);
    return encodedRedirect("error", "/sign-in", error.message);
  }

  if (!data || !data.user || !data.user.id) {
    console.error("登录成功但用户数据无效");
    await supabase.auth.signOut();
    return encodedRedirect("error", "/sign-in", "登录异常，请重试");
  }

  console.log("用户登录成功，ID:", data.user.id);

  // 登录成功后，检查用户是否为管理员
  const adminClient = createAdminClient();
  
  // 检查管理员表中是否存在该用户
  const { data: adminData, error: adminError } = await adminClient
    .from("admin_users")
    .select("*")
    .eq("user_id", data.user.id);
  
  console.log("管理员查询结果:", { adminData, adminError });

  // 如果有错误或者没有查到管理员记录
  if (adminError) {
    console.error("查询管理员记录出错:", adminError.message);
    await supabase.auth.signOut();
    return encodedRedirect("error", "/sign-in", `管理员权限查询失败: ${adminError.message}`);
  }
  
  // 检查是否有管理员记录
  if (!adminData || adminData.length === 0) {
    console.log("用户不是管理员");
    // 登出用户
    await supabase.auth.signOut();
    return encodedRedirect("error", "/sign-in", "您不是管理员，无权访问后台系统");
  }

  console.log("管理员验证成功，准备重定向到管理后台");
  // 是管理员，重定向到管理后台
  return redirect("/admin");
};

export const forgotPasswordAction = async (formData: FormData) => {
  const email = formData.get("email")?.toString();
  const supabase = await createClient();
  const origin = (await headers()).get("origin");
  const callbackUrl = formData.get("callbackUrl")?.toString();

  if (!email) {
    return encodedRedirect("error", "/forgot-password", "请填写电子邮箱");
  }

  const { error } = await supabase.auth.resetPasswordForEmail(email, {
    redirectTo: `${origin}/auth/callback?redirect_to=/protected/reset-password`,
  });

  if (error) {
    console.error(error.message);
    return encodedRedirect(
      "error",
      "/forgot-password",
      "重置密码失败，请重试"
    );
  }

  if (callbackUrl) {
    return redirect(callbackUrl);
  }

  return encodedRedirect(
    "success",
    "/forgot-password",
    "重置链接已发送，请检查您的邮箱"
  );
};

export const resetPasswordAction = async (formData: FormData) => {
  const supabase = await createClient();

  const password = formData.get("password") as string;
  const confirmPassword = formData.get("confirmPassword") as string;

  if (!password || !confirmPassword) {
    return encodedRedirect(
      "error",
      "/protected/reset-password",
      "请填写新密码和确认密码"
    );
  }

  if (password !== confirmPassword) {
    return encodedRedirect(
      "error",
      "/protected/reset-password",
      "两次输入的密码不一致"
    );
  }

  const { error } = await supabase.auth.updateUser({
    password: password,
  });

  if (error) {
    return encodedRedirect(
      "error",
      "/protected/reset-password",
      "密码更新失败"
    );
  }

  return encodedRedirect("success", "/protected/reset-password", "密码已更新");
};

export const signOutAction = async () => {
  const supabase = await createClient();
  await supabase.auth.signOut();
  return redirect("/sign-in");
};
