import SwiftUI
import Foundation

class CurrencyManager: ObservableObject {
    @Published var selectedCurrency: Currency = .rmb
    
    static let shared = CurrencyManager()
    
    private init() {
        // 从UserDefaults加载保存的货币设置
        if let savedCurrencyRawValue = UserDefaults.standard.string(forKey: "selectedCurrency"),
           let savedCurrency = Currency(rawValue: savedCurrencyRawValue) {
            selectedCurrency = savedCurrency
        }
    }
    
    func setCurrency(_ currency: Currency) {
        selectedCurrency = currency
        // 保存到UserDefaults
        UserDefaults.standard.set(currency.rawValue, forKey: "selectedCurrency")
    }
    
    func formatAmount(_ amount: Double) -> String {
        return "\(selectedCurrency.symbol) \(String(format: "%.0f", amount))"
    }
    
    func formatAmountWithDecimals(_ amount: Double) -> String {
        return "\(selectedCurrency.symbol) \(String(format: "%.2f", amount))"
    }
}

enum Currency: String, CaseIterable {
    case rmb = "人民币"
    case usd = "美元"
    case eur = "欧元"
    case jpy = "日元"
    case gbp = "英镑"
    case hkd = "港币"
    
    var symbol: String {
        switch self {
        case .rmb: return "¥"
        case .usd: return "$"
        case .eur: return "€"
        case .jpy: return "¥"
        case .gbp: return "£"
        case .hkd: return "HK$"
        }
    }
    
    var code: String {
        switch self {
        case .rmb: return "CNY"
        case .usd: return "USD"
        case .eur: return "EUR"
        case .jpy: return "JPY"
        case .gbp: return "GBP"
        case .hkd: return "HKD"
        }
    }
} 