import SwiftUI

@available(iOS 13.0, *)
struct AILoadingView: View {
    @State private var animationPhase: Double = 0
    @State private var pulseAnimation: Bool = false
    @State private var currentDots = 0
    
    let loadingTexts = [
        "AI正在分析语音内容...",
        "识别语音中的关键信息...",
        "提取金额和消费描述...",
        "匹配合适的分类..."
    ]
    
    @State private var currentTextIndex = 0
    
    var body: some View {
        VStack(spacing: 40) {
            // AI Brain Animation
            ZStack {
                // Outer glow circle
                Circle()
                    .stroke(
                        LinearGradient(
                            gradient: Gradient(colors: [.blue.opacity(0.3), .purple.opacity(0.3)]),
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        ),
                        lineWidth: 3
                    )
                    .frame(width: 120, height: 120)
                    .scaleEffect(pulseAnimation ? 1.2 : 1.0)
                    .opacity(pulseAnimation ? 0.3 : 0.8)
                
                // Main circle
                Circle()
                    .stroke(
                        LinearGradient(
                            gradient: Gradient(colors: [.blue, .purple]),
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        ),
                        lineWidth: 4
                    )
                    .frame(width: 80, height: 80)
                    .rotationEffect(.degrees(animationPhase))
                
                // AI Brain Icon
                Image(systemName: "brain.head.profile")
                    .font(.system(size: 32))
                    .foregroundStyle(
                        LinearGradient(
                            gradient: Gradient(colors: [.blue, .purple]),
                            startPoint: .topLeading,
                            endPoint: .bottomTrailing
                        )
                    )
                    .scaleEffect(pulseAnimation ? 1.1 : 1.0)
            }
            
            // Loading Text with Animated Dots
            VStack(spacing: 16) {
                Text(loadingTexts[currentTextIndex])
                    .font(.headline)
                    .foregroundColor(.primary)
                    .multilineTextAlignment(.center)
                    .animation(.easeInOut(duration: 0.5), value: currentTextIndex)
                
                // Animated dots
                HStack(spacing: 8) {
                    ForEach(0..<3, id: \.self) { index in
                        Circle()
                            .fill(Color.blue)
                            .frame(width: 8, height: 8)
                            .scaleEffect(currentDots == index ? 1.5 : 1.0)
                            .opacity(currentDots >= index ? 1.0 : 0.3)
                            .animation(.easeInOut(duration: 0.3).delay(Double(index) * 0.1), value: currentDots)
                    }
                }
            }
            
            // Progress indicator
            VStack(spacing: 8) {
                ProgressView()
                    .scaleEffect(1.2)
                    .tint(.blue)
                
                Text("请稍等，正在处理中...")
                    .font(.caption)
                    .foregroundColor(.secondary)
            }
        }
        .padding(40)
        .background(
            RoundedRectangle(cornerRadius: 20)
                .fill(.ultraThinMaterial)
                .stroke(Color.gray.opacity(0.2), lineWidth: 1)
        )
        .onAppear {
            startAnimations()
        }
    }
    
    private func startAnimations() {
        // Rotation animation
        withAnimation(.linear(duration: 3).repeatForever(autoreverses: false)) {
            animationPhase = 360
        }
        
        // Pulse animation
        withAnimation(.easeInOut(duration: 1.5).repeatForever(autoreverses: true)) {
            pulseAnimation = true
        }
        
        // Dots animation
        Timer.scheduledTimer(withTimeInterval: 0.5, repeats: true) { _ in
            withAnimation(.easeInOut(duration: 0.3)) {
                currentDots = (currentDots + 1) % 4
            }
        }
        
        // Text rotation
        Timer.scheduledTimer(withTimeInterval: 2.0, repeats: true) { _ in
            withAnimation(.easeInOut(duration: 0.5)) {
                currentTextIndex = (currentTextIndex + 1) % loadingTexts.count
            }
        }
    }
}

#Preview {
    AILoadingView()
        .padding()
} 